<?php
// File name: affiliates.php
// File location: /home/apimedi1/public_html/api/v1/routes/affiliate/affiliates.php

require_once __DIR__ . '/../../middleware/jwt.php';
require_once __DIR__ . '/../../utils/request.php';
require_once __DIR__ . '/../../handlers/Response.php';
require_once __DIR__ . '/../../../../../secure/database_router_v2.php';

$user_id = requireJWT();
$user_id_str = (string)$user_id;
$ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';

$conn = getDatabaseConnection('affiliates', 'GBL');
if (!$conn) {
    logError('Database connection failed for affiliates', 'Database', __FILE__, __LINE__);
    (new Response)->json(['error' => 'Database connection failed'], 500);
}

$conn_core = getConnMain();  // Assuming getConnMain() connects to meditagnet_meditag_core
if (!$conn_core) {
    logError('Database connection failed for core', 'Database', __FILE__, __LINE__);
    (new Response)->json(['error' => 'Database connection failed'], 500);
}

$method = $_SERVER['REQUEST_METHOD'];
$response = new Response();

switch ($method) {
    case 'POST':
        // Create new affiliate
        $input = getJsonInput();
        $role_type = $input['role_type'] ?? 'Patient';
        $entity_id = $input['entity_id'] ?? null;
        $currency = $input['currency'] ?? 'USD';
        $parent_affiliate_id = $input['parent_affiliate_id'] ?? null;
        $tier_id = $input['tier_id'] ?? null;
        $additional_notes = $input['additional_notes'] ?? null;

        // Validate required fields
        if (!in_array($role_type, ['Provider', 'Patient', 'Partner'])) {
            $response->json(['error' => 'Invalid role_type'], 400);
        }
        if ($entity_id === null) {
            $response->json(['error' => 'Missing entity_id'], 400);
        }

        // Validate entity_id exists in meditag_entities
        $stmt_core = $conn_core->prepare("SELECT entity_id FROM meditag_entities WHERE entity_id = ?");
        $stmt_core->bind_param("i", $entity_id);
        $stmt_core->execute();
        if ($stmt_core->get_result()->num_rows === 0) {
            $response->json(['error' => 'Invalid entity_id'], 400);
        }
        $stmt_core->close();

        // Check if affiliate already exists for user + role_type + entity_id
        $stmt = $conn->prepare("SELECT id FROM affiliates WHERE user_id = ? AND role_type = ? AND entity_id = ?");
        $stmt->bind_param("ssi", $user_id_str, $role_type, $entity_id);
        $stmt->execute();
        if ($stmt->get_result()->num_rows > 0) {
            $response->json(['error' => 'Affiliate already exists for this user, role, and entity'], 400);
        }
        $stmt->close();

        // Generate unique affiliate_id
        $affiliate_id = 'AFF' . strtoupper(bin2hex(random_bytes(8)));

        // Insert new affiliate
        $stmt = $conn->prepare("
            INSERT INTO affiliates 
            (user_id, affiliate_id, parent_affiliate_id, role_type, currency, tier_id, additional_notes, entity_id, recorded_by_user_id, recorded_by_role_type) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        ");
        $recorded_by_role_type = $role_type;  // Use the same role_type for recording
        $stmt->bind_param("sssssisiii", $user_id_str, $affiliate_id, $parent_affiliate_id, $role_type, $currency, $tier_id, $additional_notes, $entity_id, $user_id, $recorded_by_role_type);

        if ($stmt->execute()) {
            $new_id = $stmt->insert_id;

            // Log to audit_logs in API DB
            $conn_api = getConnAPI();
            if ($conn_api) {
                $action = 'Affiliate created';
                $resource_type = 'affiliate';
                $details = json_encode(['affiliate_id' => $affiliate_id, 'role_type' => $role_type, 'entity_id' => $entity_id, 'currency' => $currency]);
                $region_code = null;
                $stmt_api = $conn_api->prepare("
                    INSERT INTO audit_logs (user_id, action, audited_resource_type, audited_resource_id, details, ip_address, created_at, region_code) 
                    VALUES (?, ?, ?, ?, ?, ?, NOW(), ?)
                ");
                $stmt_api->bind_param("ississs", $user_id, $action, $resource_type, $new_id, $details, $ip, $region_code);
                $stmt_api->execute();
                $stmt_api->close();
            }

            http_response_code(201);
            $response->json(['affiliate_id' => $affiliate_id, 'message' => 'Affiliate created successfully']);
        } else {
            logError('Failed to insert affiliate: ' . $stmt->error, 'Database', __FILE__, __LINE__);
            $response->json(['error' => 'Failed to create affiliate'], 500);
        }
        $stmt->close();
        break;

    case 'GET':
        if (isset($_GET['affiliate_id'])) {
            $aff_id = $_GET['affiliate_id'];
            $stmt = $conn->prepare("SELECT * FROM affiliates WHERE affiliate_id = ? AND user_id = ? AND deleted_at IS NULL");
            $stmt->bind_param("ss", $aff_id, $user_id_str);
            $stmt->execute();
            $result = $stmt->get_result();
            if ($result->num_rows > 0) {
                $response->json($result->fetch_assoc());
            } else {
                $response->json(['error' => 'Affiliate not found'], 404);
            }
        } else {
            $stmt = $conn->prepare("SELECT * FROM affiliates WHERE user_id = ? AND deleted_at IS NULL");
            $stmt->bind_param("s", $user_id_str);
            $stmt->execute();
            $result = $stmt->get_result();
            $affiliates = [];
            while ($row = $result->fetch_assoc()) {
                $affiliates[] = $row;
            }
            $response->json($affiliates);
        }
        $stmt->close();
        break;

    case 'PUT':
        if (!isset($_GET['affiliate_id'])) {
            $response->json(['error' => 'Missing affiliate_id'], 400);
        }
        $aff_id = $_GET['affiliate_id'];

        // Check ownership
        $stmt = $conn->prepare("SELECT id FROM affiliates WHERE affiliate_id = ? AND user_id = ? AND deleted_at IS NULL");
        $stmt->bind_param("ss", $aff_id, $user_id_str);
        $stmt->execute();
        if ($stmt->get_result()->num_rows === 0) {
            $response->json(['error' => 'Affiliate not found or unauthorized'], 403);
        }
        $stmt->close();

        $input = getJsonInput();
        $updates = [];
        $params = [];
        $types = '';

        if (isset($input['status'])) {
            $updates[] = "status = ?";
            $params[] = $input['status'];
            $types .= 's';
        }
        if (isset($input['currency'])) {
            $updates[] = "currency = ?";
            $params[] = $input['currency'];
            $types .= 's';
        }
        // Add other updatable fields as needed (note: entity_id not updatable)

        if (empty($updates)) {
            $response->json(['error' => 'No fields to update'], 400);
        }

        $updates[] = "updated_at = CURRENT_TIMESTAMP";
        $updates[] = "updated_by_user_id = ?";
        $params[] = $user_id;
        $types .= 'i';
        $updates[] = "updated_by_role_type = ?";
        $params[] = $input['role_type'] ?? 'Patient';  // Assume or from input
        $types .= 's';
        $updates[] = "update_reason = ?";
        $params[] = $input['update_reason'] ?? 'User update';
        $types .= 's';

        $update_sql = implode(', ', $updates);
        $params[] = $aff_id;
        $types .= 's';

        $stmt = $conn->prepare("UPDATE affiliates SET $update_sql WHERE affiliate_id = ?");
        $stmt->bind_param($types, ...$params);

        if ($stmt->execute()) {
            // Audit log
            $conn_api = getConnAPI();
            if ($conn_api) {
                $action = 'Affiliate updated';
                $resource_type = 'affiliate';
                $resource_id = $aff_id;  // Use affiliate_id as string for id
                $details = json_encode($input);
                $region_code = null;
                $stmt_api = $conn_api->prepare("
                    INSERT INTO audit_logs (user_id, action, audited_resource_type, audited_resource_id, details, ip_address, created_at, region_code) 
                    VALUES (?, ?, ?, ?, ?, ?, NOW(), ?)
                ");
                $stmt_api->bind_param("ississs", $user_id, $action, $resource_type, $resource_id, $details, $ip, $region_code);
                $stmt_api->execute();
                $stmt_api->close();
            }
            $response->json(['message' => 'Affiliate updated successfully']);
        } else {
            logError('Failed to update affiliate: ' . $stmt->error, 'Database', __FILE__, __LINE__);
            $response->json(['error' => 'Failed to update affiliate'], 500);
        }
        $stmt->close();
        break;

    case 'DELETE':
        if (!isset($_GET['affiliate_id'])) {
            $response->json(['error' => 'Missing affiliate_id'], 400);
        }
        $aff_id = $_GET['affiliate_id'];

        // Check ownership
        $stmt = $conn->prepare("SELECT id FROM affiliates WHERE affiliate_id = ? AND user_id = ? AND deleted_at IS NULL");
        $stmt->bind_param("ss", $aff_id, $user_id_str);
        $stmt->execute();
        $row = $stmt->get_result()->fetch_assoc();
        if (!$row) {
            $response->json(['error' => 'Affiliate not found or unauthorized'], 403);
        }
        $resource_id = $row['id'];
        $stmt->close();

        // Soft delete
        $input = getJsonInput();
        $delete_reason = $input['delete_reason'] ?? 'User requested deletion';
        $deleted_by_role_type = 'Patient';  // Assume
        $stmt = $conn->prepare("
            UPDATE affiliates SET deleted_at = CURRENT_TIMESTAMP, delete_reason = ?, deleted_by_user_id = ?, deleted_by_role_type = ? 
            WHERE affiliate_id = ?
        ");
        $stmt->bind_param("siss", $delete_reason, $user_id, $deleted_by_role_type, $aff_id);

        if ($stmt->execute()) {
            // Audit log
            $conn_api = getConnAPI();
            if ($conn_api) {
                $action = 'Affiliate deleted';
                $resource_type = 'affiliate';
                $details = json_encode(['delete_reason' => $delete_reason]);
                $region_code = null;
                $stmt_api = $conn_api->prepare("
                    INSERT INTO audit_logs (user_id, action, audited_resource_type, audited_resource_id, details, ip_address, created_at, region_code) 
                    VALUES (?, ?, ?, ?, ?, ?, NOW(), ?)
                ");
                $stmt_api->bind_param("ississs", $user_id, $action, $resource_type, $resource_id, $details, $ip, $region_code);
                $stmt_api->execute();
                $stmt_api->close();
            }
            $response->json(['message' => 'Affiliate deleted successfully']);
        } else {
            logError('Failed to delete affiliate: ' . $stmt->error, 'Database', __FILE__, __LINE__);
            $response->json(['error' => 'Failed to delete affiliate'], 500);
        }
        $stmt->close();
        break;

    default:
        $response->json(['error' => 'Method not allowed'], 405);
}
?>