<?php
// File name: profile.php
// File location: /home/apimedi1/public_html/api/v1/routes/profile.php
require_once __DIR__ . '/../../../../secure/config_v2.php';
require_once __DIR__ . '/../../../../secure/database_router_v2.php';
require_once __DIR__ . '/../middleware/jwt.php';
require_once __DIR__ . '/../utils/request.php';
require_once __DIR__ . '/../handlers/Response.php';
// Manually validate JWT to access full payload
$header = $_SERVER['HTTP_AUTHORIZATION'] ?? '';
if (!preg_match('/Bearer\s+(\S+)/', $header, $matches)) {
    (new Response)->json(['error' => 'Missing token'], 401);
}
$payload = validateDeviceOrNormalToken($matches[1]);
if (!$payload) {
    (new Response)->json(['error' => 'Invalid token'], 401);
}
$userId = $payload['userId'];
$regionCode = $payload['regionCode'] ?? 'NA';

// === NEW: Update api_requests with user_id after successful authentication ===
$conn_api = getConnAPI();
if ($conn_api && defined('REQUEST_ID')) {
    $request_id = REQUEST_ID;
    $update_stmt = $conn_api->prepare("UPDATE api_requests SET user_id = ? WHERE request_id = ?");
    if ($update_stmt) {
        $update_stmt->bind_param("is", $userId, $request_id);
        $update_stmt->execute();
        $update_stmt->close();
    }
}
// === End of new code ===

// Connect to regional main DB
$connRegional = getDatabaseConnection('main', $regionCode);
if (!$connRegional) {
    (new Response)->json(['error' => 'Database connection failed'], 500);
}
// Connect to global main DB for partner profile pics
$connGlobal = getDatabaseConnection('main', 'GBL');
if (!$connGlobal) {
    (new Response)->json(['error' => 'Global database connection failed'], 500);
}
// Helper function to fetch profile data
function fetchProfile($connRegional, $connGlobal, $userId) {
    // Fetch basic user info from users table (removed account_status as it's not in regional schema)
    $stmtUser = $connRegional->prepare("
        SELECT first_name, middle_name, last_name, email, mobile_telephone, title,
               profile_photo, verification_stage, base_currency, timezone,
               updated_at
        FROM users WHERE id = ?
    ");
    if (!$stmtUser) {
        error_log("Prepare failed: " . $connRegional->error);
        return null;
    }
    $stmtUser->bind_param("i", $userId);
    $stmtUser->execute();
    $userResult = $stmtUser->get_result()->fetch_assoc();
    $stmtUser->close();
    if (!$userResult) {
        return null;
    }
    $fullName = trim(($userResult['title'] ?? '') . ' ' . ($userResult['first_name'] ?? '') . ' ' . ($userResult['middle_name'] ?? '') . ' ' . ($userResult['last_name'] ?? ''));
    // Fetch all roles' data
    $roles = [];
    $patientId = null;
    $providerId = null;
    $partnerIds = [];
    $profilePics = [];
    // Patient
    $stmtPatient = $connRegional->prepare("
        SELECT id, profile_pic, username, bio, medical_aid_name, medical_aid_number, medical_aid_plan
        FROM patients WHERE user_id = ?
        ORDER BY CASE WHEN username IS NOT NULL AND username != '' THEN 1 ELSE 0 END DESC, id DESC LIMIT 1
    ");
    if ($stmtPatient) {
        $stmtPatient->bind_param("i", $userId);
        $stmtPatient->execute();
        $patientResult = $stmtPatient->get_result()->fetch_assoc();
        $stmtPatient->close();
        if ($patientResult) {
            $patientId = $patientResult['id'];
            $patientProfilePic = $patientResult['profile_pic'] ?? '/images/icons/user-placeholder.png';
            // Special check for default logo to fallback to placeholder
            if (strpos($patientProfilePic, 'logo_dark.png') !== false) {
                $patientProfilePic = '/images/icons/user-placeholder.png';
            }
            $profilePics['patient'] = $patientProfilePic;
            $roles['Patient'] = [
                'patientId' => $patientId,
                'username' => $patientResult['username'] ?? '',
                'bio' => $patientResult['bio'] ?? '',
                'medicalAidName' => $patientResult['medical_aid_name'] ?? null,
                'medicalAidNumber' => $patientResult['medical_aid_number'] ?? null,
                'medicalAidPlan' => $patientResult['medical_aid_plan'] ?? null,
                'profilePic' => $profilePics['patient']
            ];
        }
    }
    // Provider
    $stmtProvider = $connRegional->prepare("
        SELECT id, profile_pic, username, bio, profession, health_council, health_council_reg_number
        FROM providers WHERE user_id = ?
        ORDER BY CASE WHEN username IS NOT NULL AND username != '' THEN 1 ELSE 0 END DESC, id DESC LIMIT 1
    ");
    if ($stmtProvider) {
        $stmtProvider->bind_param("i", $userId);
        $stmtProvider->execute();
        $providerResult = $stmtProvider->get_result()->fetch_assoc();
        $stmtProvider->close();
        if ($providerResult) {
            $providerId = $providerResult['id'];
            $profilePics['provider'] = $providerResult['profile_pic'] ?? '/images/icons/user-placeholder.png';
            $roles['Provider'] = [
                'providerId' => $providerId,
                'username' => $providerResult['username'] ?? '',
                'bio' => $providerResult['bio'] ?? '',
                'profession' => $providerResult['profession'] ?? null,
                'healthCouncil' => $providerResult['health_council'] ?? null,
                'healthCouncilRegNumber' => $providerResult['health_council_reg_number'] ?? null,
                'profilePic' => $profilePics['provider']
            ];
            // Fetch qualifications
            $qualifications = [];
            $stmtQual = $connRegional->prepare("
                SELECT id, institution_name, qualification_name, field_of_study, date_obtained,
                       certificate_number, issuing_authority, verification_status
                FROM provider_qualifications WHERE provider_id = ? AND is_active = 1 AND deleted_at IS NULL
            ");
            if ($stmtQual) {
                $stmtQual->bind_param("i", $providerId);
                $stmtQual->execute();
                $qualResult = $stmtQual->get_result();
                while ($qual = $qualResult->fetch_assoc()) {
                    $qualifications[] = $qual;
                }
                $stmtQual->close();
                $roles['Provider']['qualifications'] = $qualifications;
            }
        }
    }
    // Partners (multiple)
    $partners = [];
    $stmtPartner = $connRegional->prepare("
        SELECT id, username, bio, company_name, registration_number, vat_number
        FROM partners WHERE user_id = ?
    ");
    if ($stmtPartner) {
        $stmtPartner->bind_param("i", $userId);
        $stmtPartner->execute();
        $partnerResult = $stmtPartner->get_result();
        while ($partner = $partnerResult->fetch_assoc()) {
            $partnerId = $partner['id'];
            $partnerIds[] = $partnerId;
            $partners[] = [
                'partnerId' => $partnerId,
                'username' => $partner['username'] ?? '',
                'bio' => $partner['bio'] ?? '',
                'companyName' => $partner['company_name'] ?? null,
                'registrationNumber' => $partner['registration_number'] ?? null,
                'vatNumber' => $partner['vat_number'] ?? null,
                'profilePic' => '' // Placeholder, to be filled from global
            ];
        }
        $stmtPartner->close();
        // Fetch profile pics from global DB for each partner
        foreach ($partners as &$p) {
            $pid = $p['partnerId'];
            $stmtPic = $connGlobal->prepare("
                SELECT profile_pic FROM partners WHERE id = ?
            ");
            if ($stmtPic) {
                $stmtPic->bind_param("i", $pid);
                $stmtPic->execute();
                $picResult = $stmtPic->get_result()->fetch_assoc();
                $p['profilePic'] = $picResult['profile_pic'] ?? '/images/icons/user-placeholder.png';
                $profilePics['partners'][] = [
                    'partnerId' => $pid,
                    'profilePic' => $p['profilePic']
                ];
                $stmtPic->close();
            }
        }
        unset($p); // Unset reference
        if (!empty($partners)) {
            $roles['Partner'] = $partners;
        }
    }
    // Fetch addresses
    $addresses = [];
    $stmtAddr = $connRegional->prepare("
        SELECT id, address_type AS type, address_line_1 AS line1, address_line_2 AS line2,
               city, region, country, zip, effective_date AS effectiveDate, is_current AS isCurrent
        FROM user_addresses WHERE user_id = ? AND deleted_at IS NULL ORDER BY effective_date DESC
    ");
    if ($stmtAddr) {
        $stmtAddr->bind_param("i", $userId);
        $stmtAddr->execute();
        $addrResult = $stmtAddr->get_result();
        while ($addr = $addrResult->fetch_assoc()) {
            $addresses[] = $addr;
        }
        $stmtAddr->close();
    }
    // Fetch identifications
    $identifications = [];
    $stmtId = $connRegional->prepare("
        SELECT id, id_type AS type, id_value AS value, issued_date AS issuedDate, expiry_date AS expiryDate,
               issuing_country AS issuingCountry, issuing_authority, status, validation_status AS validationStatus
        FROM user_identifications WHERE user_id = ? AND deleted_at IS NULL ORDER BY issued_date DESC
    ");
    if ($stmtId) {
        $stmtId->bind_param("i", $userId);
        $stmtId->execute();
        $idResult = $stmtId->get_result();
        while ($id = $idResult->fetch_assoc()) {
            $identifications[] = $id;
        }
        $stmtId->close();
    }
    return [
        'userId' => $userId,
        'fullName' => $fullName,
        'email' => $userResult['email'],
        'mobileTelephone' => $userResult['mobile_telephone'],
        'verificationStage' => $userResult['verification_stage'] ?? null,
        'accountStatus' => 'active', // Hardcoded fallback
        'baseCurrency' => $userResult['base_currency'] ?? 'USD',
        'timezone' => $userResult['timezone'] ?? 'UTC',
        'patientId' => $patientId,
        'providerId' => $providerId,
        'partnerIds' => $partnerIds,
        'profilePics' => $profilePics,
        'addresses' => $addresses,
        'identifications' => $identifications,
        'roles' => $roles
    ];
}
// GET /profiles - Fetch profile
if ($_SERVER['REQUEST_METHOD'] === 'GET') {
    $profile = fetchProfile($connRegional, $connGlobal, $userId);
    if ($profile) {
        (new Response)->json($profile);
    } else {
        (new Response)->json(['error' => 'Profile not found'], 404);
    }
}
// PUT /profiles - Update profile (partial)
if ($_SERVER['REQUEST_METHOD'] === 'PUT') {
    $data = getJsonInput();
    // Update users table
    $updateUserSql = "UPDATE users SET ";
    $updateUserParams = [];
    $updateUserTypes = '';
    if (isset($data['firstName'])) {
        $updateUserSql .= "first_name = ?, ";
        $updateUserParams[] = $data['firstName'];
        $updateUserTypes .= 's';
    }
    if (isset($data['middleName'])) {
        $updateUserSql .= "middle_name = ?, ";
        $updateUserParams[] = $data['middleName'];
        $updateUserTypes .= 's';
    }
    if (isset($data['lastName'])) {
        $updateUserSql .= "last_name = ?, ";
        $updateUserParams[] = $data['lastName'];
        $updateUserTypes .= 's';
    }
    if (isset($data['email'])) {
        $updateUserSql .= "email = ?, ";
        $updateUserParams[] = $data['email'];
        $updateUserTypes .= 's';
    }
    if (isset($data['mobileTelephone'])) {
        $updateUserSql .= "mobile_telephone = ?, ";
        $updateUserParams[] = $data['mobileTelephone'];
        $updateUserTypes .= 's';
    }
    if (substr($updateUserSql, -2) === ', ') {
        $updateUserSql = substr($updateUserSql, 0, -2);
    }
    $updateUserSql .= " WHERE id = ?";
    $updateUserParams[] = $userId;
    $updateUserTypes .= 'i';
    if (count($updateUserParams) > 1) { // If any fields to update
        $stmtUser = $connRegional->prepare($updateUserSql);
        if (!$stmtUser) {
            error_log("Prepare failed for users update: " . $connRegional->error);
            (new Response)->json(['error' => 'Update failed'], 500);
        }
        $stmtUser->bind_param($updateUserTypes, ...$updateUserParams);
        $stmtUser->execute();
        $stmtUser->close();
    }
    // Update role-specific (example for Patient; extend based on roleType)
    $updateRoleSql = "UPDATE patients SET ";
    $updateRoleParams = [];
    $updateRoleTypes = '';
    if (isset($data['profilePic'])) {
        $updateRoleSql .= "profile_pic = ?, ";
        $updateRoleParams[] = $data['profilePic'];
        $updateRoleTypes .= 's';
    }
    if (isset($data['username'])) {
        $updateRoleSql .= "username = ?, ";
        $updateRoleParams[] = $data['username'];
        $updateRoleTypes .= 's';
    }
    if (isset($data['bio'])) {
        $updateRoleSql .= "bio = ?, ";
        $updateRoleParams[] = $data['bio'];
        $updateRoleTypes .= 's';
    }
    if (substr($updateRoleSql, -2) === ', ') {
        $updateRoleSql = substr($updateRoleSql, 0, -2);
    }
    $updateRoleSql .= " WHERE user_id = ?";
    $updateRoleParams[] = $userId;
    $updateRoleTypes .= 'i';
    if (count($updateRoleParams) > 1) {
        $stmtRole = $connRegional->prepare($updateRoleSql);
        if (!$stmtRole) {
            error_log("Prepare failed for patients update: " . $connRegional->error);
            (new Response)->json(['error' => 'Update failed'], 500);
        }
        $stmtRole->bind_param($updateRoleTypes, ...$updateRoleParams);
        $stmtRole->execute();
        $stmtRole->close();
    }
    // Handle addresses (add/update)
    if (isset($data['addresses']) && is_array($data['addresses'])) {
        foreach ($data['addresses'] as $addr) {
            $addrId = $addr['id'] ?? null;
            $type = $addr['type'] ?? null;
            $line1 = $addr['line1'] ?? null;
            $line2 = $addr['line2'] ?? null;
            $city = $addr['city'] ?? null;
            $region = $addr['region'] ?? null;
            $country = $addr['country'] ?? null;
            $zip = $addr['zip'] ?? null;
            $effectiveDate = $addr['effectiveDate'] ?? date('Y-m-d');
            if (!$type) continue;
            $userRoleType = 'Patient'; // Determine dynamically if needed
            if ($addrId) {
                // Update
                $sql = "UPDATE user_addresses SET address_type = ?, address_line_1 = ?, address_line_2 = ?, city = ?, region = ?, country = ?, zip = ?, effective_date = ?, updated_at = NOW() WHERE id = ?";
                $stmt = $connRegional->prepare($sql);
                if (!$stmt) {
                    error_log("Prepare failed for address update: " . $connRegional->error);
                    continue;
                }
                $stmt->bind_param("ssssssssi", $type, $line1, $line2, $city, $region, $country, $zip, $effectiveDate, $addrId);
            } else {
                // Insert
                $sql = "INSERT INTO user_addresses (user_id, user_role_type, address_type, address_line_1, address_line_2, city, region, country, zip, effective_date, is_current, created_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 1, NOW())";
                $stmt = $connRegional->prepare($sql);
                if (!$stmt) {
                    error_log("Prepare failed for address insert: " . $connRegional->error);
                    continue;
                }
                $stmt->bind_param("isssssssss", $userId, $userRoleType, $type, $line1, $line2, $city, $region, $country, $zip, $effectiveDate);
            }
            $stmt->execute();
            $stmt->close();
        }
    }
    // Handle identifications (example without encryption; add if needed)
    if (isset($data['identifications']) && is_array($data['identifications'])) {
        foreach ($data['identifications'] as $idData) {
            $idId = $idData['id'] ?? null;
            $type = $idData['type'] ?? null;
            $value = $idData['value'] ?? null; // Encrypt if necessary
            $issuedDate = $idData['issuedDate'] ?? null;
            $expiryDate = $idData['expiryDate'] ?? null;
            $issuingCountry = $idData['issuingCountry'] ?? null;
            $issuingAuthority = $idData['issuingAuthority'] ?? null;
            if (!$type || !$value) continue;
            if ($idId) {
                // Update
                $sql = "UPDATE user_identifications SET id_type = ?, id_value = ?, issued_date = ?, expiry_date = ?, issuing_country = ?, issuing_authority = ?, updated_at = NOW() WHERE id = ?";
                $stmt = $connRegional->prepare($sql);
                if (!$stmt) {
                    error_log("Prepare failed for id update: " . $connRegional->error);
                    continue;
                }
                $stmt->bind_param("ssssssi", $type, $value, $issuedDate, $expiryDate, $issuingCountry, $issuingAuthority, $idId);
            } else {
                // Insert
                $sql = "INSERT INTO user_identifications (user_id, id_type, id_value, issued_date, expiry_date, issuing_country, issuing_authority, status, validation_status, system_created_at, recorded_at) VALUES (?, ?, ?, ?, ?, ?, ?, 'active', 'Pending Validation', NOW(), NOW())";
                $stmt = $connRegional->prepare($sql);
                if (!$stmt) {
                    error_log("Prepare failed for id insert: " . $connRegional->error);
                    continue;
                }
                $stmt->bind_param("issssss", $userId, $type, $value, $issuedDate, $expiryDate, $issuingCountry, $issuingAuthority);
            }
            $stmt->execute();
            $stmt->close();
        }
    }
    // Role-specific updates (Provider qualifications example)
    if (isset($data['roleSpecific']['qualifications']) && is_array($data['roleSpecific']['qualifications'])) {
        // Get providerId from roleSpecific or fetch
        $providerId = $data['roleSpecific']['providerId'] ?? 0; // Assume known
        foreach ($data['roleSpecific']['qualifications'] as $qual) {
            $qualId = $qual['id'] ?? null;
            $institution = $qual['institutionName'] ?? null;
            $qualification = $qual['qualificationName'] ?? null;
            $field = $qual['fieldOfStudy'] ?? null;
            $dateObtained = $qual['dateObtained'] ?? null;
            $certNum = $qual['certificateNumber'] ?? null;
            $authority = $qual['issuingAuthority'] ?? null;
            if (!$institution || !$qualification || !$dateObtained) continue;
            if ($qualId) {
                // Update
                $sql = "UPDATE provider_qualifications SET institution_name = ?, qualification_name = ?, field_of_study = ?, date_obtained = ?, certificate_number = ?, issuing_authority = ?, updated_at = NOW() WHERE id = ?";
                $stmt = $connRegional->prepare($sql);
                $stmt->bind_param("ssssssi", $institution, $qualification, $field, $dateObtained, $certNum, $authority, $qualId);
            } else {
                // Insert
                $sql = "INSERT INTO provider_qualifications (user_id, provider_id, institution_name, qualification_name, field_of_study, date_obtained, certificate_number, issuing_authority, verification_status, created_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?, 'Pending', NOW())";
                $stmt = $connRegional->prepare($sql);
                $stmt->bind_param("iissssss", $userId, $providerId, $institution, $qualification, $field, $dateObtained, $certNum, $authority);
            }
            $stmt->execute();
            $stmt->close();
        }
    }
    $updatedProfile = fetchProfile($connRegional, $connGlobal, $userId);
    (new Response)->json(['message' => 'Profile updated successfully', 'updatedProfile' => $updatedProfile]);
}
// POST /profiles/addresses - Add new address
if ($_SERVER['REQUEST_METHOD'] === 'POST' && preg_match('#^profiles/addresses$#', $uri)) {
    $data = getJsonInput();
    $type = $data['type'] ?? null;
    $line1 = $data['line1'] ?? null;
    $line2 = $data['line2'] ?? null;
    $city = $data['city'] ?? null;
    $region = $data['region'] ?? null;
    $country = $data['country'] ?? null;
    $zip = $data['zip'] ?? null;
    $effectiveDate = $data['effectiveDate'] ?? date('Y-m-d');
    if (!$type) {
        (new Response)->json(['error' => 'Type required'], 400);
    }
    $userRoleType = 'Patient'; // Determine dynamically if needed
    $sql = "INSERT INTO user_addresses (user_id, user_role_type, address_type, address_line_1, address_line_2, city, region, country, zip, effective_date, is_current, created_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 1, NOW())";
    $stmt = $connRegional->prepare($sql);
    if (!$stmt) {
        error_log("Prepare failed for address insert: " . $connRegional->error);
        (new Response)->json(['error' => 'Add failed'], 500);
    }
    $stmt->bind_param("isssssssss", $userId, $userRoleType, $type, $line1, $line2, $city, $region, $country, $zip, $effectiveDate);
    if ($stmt->execute()) {
        (new Response)->json(['message' => 'Address added', 'addressId' => $stmt->insert_id], 201);
    } else {
        error_log("Execute failed for address insert: " . $stmt->error);
        (new Response)->json(['error' => 'Add failed'], 500);
    }
    $stmt->close();
}
// DELETE /profiles/addresses/{addressId} - Soft delete address
if ($_SERVER['REQUEST_METHOD'] === 'DELETE' && preg_match('#^profiles/addresses/(\d+)$#', $uri, $m)) {
    $addrId = $m[1];
    $data = getJsonInput();
    $deleteReason = $data['deleteReason'] ?? 'User request';
    $sql = "UPDATE user_addresses SET deleted_at = NOW(), delete_reason = ? WHERE id = ? AND user_id = ?";
    $stmt = $connRegional->prepare($sql);
    if (!$stmt) {
        error_log("Prepare failed for address delete: " . $connRegional->error);
        (new Response)->json(['error' => 'Delete failed'], 500);
    }
    $stmt->bind_param("sii", $deleteReason, $addrId, $userId);
    if ($stmt->execute() && $stmt->affected_rows > 0) {
        (new Response)->json(['message' => 'Address deleted']);
    } else {
        (new Response)->json(['error' => 'Delete failed or not found'], 404);
    }
    $stmt->close();
}
// Similar for identifications and other endpoints...
// Close connection
$connRegional->close();
$connGlobal->close();
?>