<?php
// File name: prescriptions.php
// File location: /home/apimedi1/public_html/api/v1/routes/prescription/prescriptions.php

require_once __DIR__ . '/../../utils/jwt.php';
require_once __DIR__ . '/../../utils/region.php';
require_once __DIR__ . '/../../utils/request.php';
require_once __DIR__ . '/../../handlers/Response.php';
require_once __DIR__ . '/../../middleware/jwt.php';

requireJWT();

$response = new Response();
$method = $_SERVER['REQUEST_METHOD'];
$uri = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH); // For detecting /validate

$is_validate = (strpos($uri, '/validate') !== false);

if (isset($_GET['patient_id'])) {
    // Get all prescriptions for patient (GET only)
    if ($method !== 'GET') {
        $response->json(['error' => 'Method not allowed'], 405);
    }

    $patient_id = (int)$_GET['patient_id'];
    $region = getPatientRegion($patient_id);
    $conn = getDatabaseConnection('prescriptions', $region);

    if (!$conn) {
        $response->json(['error' => 'Database connection failed'], 500);
    }

    $stmt = $conn->prepare("SELECT * FROM prescriptions WHERE patient_id = ? AND deleted_at IS NULL");
    $stmt->bind_param("i", $patient_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $prescriptions = [];
    while ($row = $result->fetch_assoc()) {
        $prescriptions[] = $row;
    }
    $response->json($prescriptions);

} elseif (isset($_GET['prescription_id'])) {
    // Handle single prescription (GET, PUT, DELETE, and PUT /validate)
    $prescription_id = (int)$_GET['prescription_id'];
    $regions = ['NA', 'ZA', 'US'];
    $prescription = null;
    $conn = null;
    $region = null;

    foreach ($regions as $r) {
        $conn_temp = getDatabaseConnection('prescriptions', $r);
        if (!$conn_temp) continue;

        $stmt = $conn_temp->prepare("SELECT * FROM prescriptions WHERE prescription_id = ? AND deleted_at IS NULL");
        $stmt->bind_param("i", $prescription_id);
        $stmt->execute();
        $result = $stmt->get_result();
        if ($row = $result->fetch_assoc()) {
            $prescription = $row;
            $conn = $conn_temp;
            $region = $r;
            break;
        }
    }

    if (!$prescription) {
        $response->json(['error' => 'Prescription not found'], 404);
    }

    if ($method === 'GET') {
        // Fetch items
        $stmt = $conn->prepare("SELECT * FROM prescription_items WHERE prescription_id = ? AND deleted_at IS NULL");
        $stmt->bind_param("i", $prescription_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $items = [];
        while ($row = $result->fetch_assoc()) {
            // Fetch linked diagnoses
            $item_id = $row['prescription_item_id'];
            $stmt_link = $conn->prepare("SELECT diagnosis_id FROM prescription_diagnosis_link WHERE prescription_item_id = ? AND deleted_at IS NULL");
            $stmt_link->bind_param("i", $item_id);
            $stmt_link->execute();
            $link_result = $stmt_link->get_result();
            $diagnosis_ids = [];
            while ($link_row = $link_result->fetch_assoc()) {
                $diagnosis_ids[] = $link_row['diagnosis_id'];
            }
            $row['diagnosis_ids'] = $diagnosis_ids;
            $items[] = $row;
        }
        $prescription['items'] = $items;
        $response->json($prescription);

    } elseif ($method === 'PUT' && !$is_validate) {
        $input = getJsonInput();
        $conn->begin_transaction();
        try {
            $updated = false;
            if (isset($input['notes'])) {
                $stmt = $conn->prepare("UPDATE prescriptions SET notes = ?, updated_at = NOW(), update_reason = 'API update' WHERE prescription_id = ?");
                $stmt->bind_param("si", $input['notes'], $prescription_id);
                $stmt->execute();
                $updated = true;
            }
            if (isset($input['validation_status'])) {
                // Validate enum
                if (!in_array($input['validation_status'], ['Validated', 'Pending Validation', 'Rejected'])) {
                    throw new Exception('Invalid validation_status');
                }
                $validation_reason = isset($input['validation_reason']) ? $input['validation_reason'] : null;
                $validated_by = $GLOBALS['auth_user_id'];
                $stmt = $conn->prepare("UPDATE prescriptions SET validation_status = ?, validation_reason = ?, validated_by_user_id = ?, validated_at = NOW() WHERE prescription_id = ?");
                $stmt->bind_param("ssii", $input['validation_status'], $validation_reason, $validated_by, $prescription_id);
                $stmt->execute();
                $updated = true;
            }
            // Add logic for updating items if needed

            if ($updated) {
                // Log history
                $change_type = 'modified';
                $change_notes = 'API update';
                $changed_by = $GLOBALS['auth_user_id'];
                $stmt_hist = $conn->prepare("INSERT INTO prescription_history (prescription_id, change_type, change_notes, changed_by) VALUES (?, ?, ?, ?)");
                $stmt_hist->bind_param("issi", $prescription_id, $change_type, $change_notes, $changed_by);
                $stmt_hist->execute();
            }

            $conn->commit();
            $response->json(['success' => true]);
        } catch (Exception $e) {
            $conn->rollback();
            $response->json(['error' => $e->getMessage()], 500);
        }

    } elseif ($method === 'PUT' && $is_validate) {
        $input = getJsonInput();
        if (!isset($input['validation_status']) || !in_array($input['validation_status'], ['Validated', 'Pending Validation', 'Rejected'])) {
            $response->json(['error' => 'Invalid or missing validation_status'], 400);
        }
        $conn->begin_transaction();
        try {
            $validation_reason = isset($input['validation_reason']) ? $input['validation_reason'] : null;
            $validated_by = $GLOBALS['auth_user_id'];
            $stmt = $conn->prepare("UPDATE prescriptions SET validation_status = ?, validation_reason = ?, validated_by_user_id = ?, validated_at = NOW() WHERE prescription_id = ?");
            $stmt->bind_param("ssii", $input['validation_status'], $validation_reason, $validated_by, $prescription_id);
            $stmt->execute();

            // Log history
            $change_type = 'modified';
            $change_notes = 'Validation update: ' . $input['validation_status'];
            $changed_by = $validated_by;
            $stmt_hist = $conn->prepare("INSERT INTO prescription_history (prescription_id, change_type, change_notes, changed_by) VALUES (?, ?, ?, ?)");
            $stmt_hist->bind_param("issi", $prescription_id, $change_type, $change_notes, $changed_by);
            $stmt_hist->execute();

            $conn->commit();
            $response->json(['success' => true]);
        } catch (Exception $e) {
            $conn->rollback();
            $response->json(['error' => $e->getMessage()], 500);
        }

    } elseif ($method === 'DELETE') {
        $conn->begin_transaction();
        try {
            $stmt = $conn->prepare("UPDATE prescriptions SET deleted_at = NOW(), delete_reason = 'API delete' WHERE prescription_id = ?");
            $stmt->bind_param("i", $prescription_id);
            $stmt->execute();

            // Log history
            $change_type = 'canceled';
            $change_notes = 'API delete';
            $changed_by = $GLOBALS['auth_user_id'];
            $stmt_hist = $conn->prepare("INSERT INTO prescription_history (prescription_id, change_type, change_notes, changed_by) VALUES (?, ?, ?, ?)");
            $stmt_hist->bind_param("issi", $prescription_id, $change_type, $change_notes, $changed_by);
            $stmt_hist->execute();

            $conn->commit();
            $response->json(['success' => true]);
        } catch (Exception $e) {
            $conn->rollback();
            $response->json(['error' => $e->getMessage()], 500);
        }

    } else {
        $response->json(['error' => 'Method not allowed'], 405);
    }

} else {
    // Create new prescription (POST only)
    if ($method !== 'POST') {
        $response->json(['error' => 'Method not allowed'], 405);
    }

    $input = getJsonInput();
    if (!isset($input['patient_id'])) {
        $response->json(['error' => 'Missing patient_id'], 400);
    }

    $patient_id = (int)$input['patient_id'];
    $region = getPatientRegion($patient_id);
    $conn = getDatabaseConnection('prescriptions', $region);

    if (!$conn) {
        $response->json(['error' => 'Database connection failed'], 500);
    }

    $conn->begin_transaction();
    try {
        $key = bin2hex(random_bytes(32));
        $provider_id = $GLOBALS['auth_user_id'];
        $consultation_id = isset($input['consultation_id']) ? (int)$input['consultation_id'] : null;
        $notes = isset($input['notes']) ? $input['notes'] : null;

        $stmt = $conn->prepare("INSERT INTO prescriptions (patient_id, provider_id, consultation_id, notes, `key`) VALUES (?, ?, ?, ?, ?)");
        $stmt->bind_param("iiiss", $patient_id, $provider_id, $consultation_id, $notes, $key);
        $stmt->execute();
        $prescription_id = $conn->insert_id;

        // Log history
        $change_type = 'created';
        $change_notes = 'API creation';
        $changed_by = $provider_id;
        $stmt_hist = $conn->prepare("INSERT INTO prescription_history (prescription_id, change_type, change_notes, changed_by) VALUES (?, ?, ?, ?)");
        $stmt_hist->bind_param("issi", $prescription_id, $change_type, $change_notes, $changed_by);
        $stmt_hist->execute();

        if (isset($input['items']) && is_array($input['items'])) {
            foreach ($input['items'] as $item) {
                if (!isset($item['drug_id']) || !isset($item['form_id']) || !isset($item['units'])) {
                    throw new Exception('Missing required item fields');
                }
                $dosage = isset($item['dosage']) ? (float)$item['dosage'] : null;
                $frequency = isset($item['frequency']) ? $item['frequency'] : null;
                $route = isset($item['route']) ? $item['route'] : null;
                $duration = isset($item['duration']) ? $item['duration'] : null;
                $duration_unit = isset($item['duration_unit']) ? $item['duration_unit'] : 'days';
                $instructions = isset($item['instructions']) ? $item['instructions'] : null;

                $stmt_item = $conn->prepare("INSERT INTO prescription_items (prescription_id, drug_id, form_id, dosage, units, frequency, route, duration, duration_unit, instructions) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
                $stmt_item->bind_param("iiidssssss", $prescription_id, $item['drug_id'], $item['form_id'], $dosage, $item['units'], $frequency, $route, $duration, $duration_unit, $instructions);
                $stmt_item->execute();
                $item_id = $conn->insert_id;

                // Link to diagnoses if provided
                if (isset($item['diagnosis_ids']) && is_array($item['diagnosis_ids'])) {
                    foreach ($item['diagnosis_ids'] as $diag_id) {
                        $stmt_link = $conn->prepare("INSERT INTO prescription_diagnosis_link (prescription_item_id, diagnosis_id) VALUES (?, ?)");
                        $stmt_link->bind_param("ii", $item_id, $diag_id);
                        $stmt_link->execute();
                    }
                }
            }
        }

        $conn->commit();
        $response->json(['prescription_id' => $prescription_id]);
    } catch (Exception $e) {
        $conn->rollback();
        $response->json(['error' => $e->getMessage()], 500);
    }
}
?>