<?php
// File name: family_history.php
// File location: /home/apimedi1/public_html/api/v1/routes/medical/family_history.php

require_once __DIR__ . '/../../middleware/jwt.php';

$userId = requireJWT();
$patientId = (int)($_GET['patient_id'] ?? ($uri_parts[2] ?? 0));
if ($patientId <= 0) {
    (new Response)->json(['error' => 'Invalid patient ID'], 400);
}

$region = getPatientRegion($patientId);
$conn = getDatabaseConnection('medical_data', $region);

// Fetch all family members
$stmt = $conn->prepare("
    SELECT 
        fm.id,
        fm.relationship_id,
        fm.first_name,
        fm.last_name,
        fm.date_of_birth,
        fm.gender,
        fm.is_deceased,
        fm.age_at_death,
        fm.date_of_death,
        fm.validation_status,
        fm.validated_by_user_id,
        fm.validated_by_role_type,
        fm.validated_at,
        fm.validation_reason,
        fm.recorded_by_user_id,
        fm.recorded_by_role_type,
        fm.recorded_at,
        fm.updated_by_user_id,
        fm.updated_by_role_type,
        fm.updated_at,
        fm.update_reason,
        -- Additional details
        fmd.highest_qualification,
        fmd.years_of_uninterrupted_employment,
        fmd.current_salary,
        fmd.salary_currency,
        fmd.total_debt,
        fmd.debt_currency,
        fmd.type_of_house,
        fmd.number_of_people_in_household,
        fmd.number_of_wives,
        fmd.number_of_children,
        fmd.is_breadwinner,
        fmd.notes AS details_notes
    FROM family_members fm
    LEFT JOIN family_member_details fmd ON fm.id = fmd.family_member_id AND fmd.deleted_at IS NULL
    WHERE fm.patient_id = ?
      AND fm.deleted_at IS NULL
    ORDER BY fm.id
");
if (!$stmt) {
    (new Response)->json(['error' => 'Database query failed: ' . $conn->error], 500);
}

$stmt->bind_param("i", $patientId);
$stmt->execute();
$result = $stmt->get_result();

$family = [];
while ($member = $result->fetch_assoc()) {
    $member_id = $member['id'];

    // Format dates
    $member['date_of_birth'] = $member['date_of_birth'] ? date('Y-m-d', strtotime($member['date_of_birth'])) : null;
    $member['date_of_death'] = $member['date_of_death'] ? date('Y-m-d', strtotime($member['date_of_death'])) : null;
    $member['validated_at'] = $member['validated_at'] ? date('Y-m-d H:i:s', strtotime($member['validated_at'])) : null;
    $member['recorded_at'] = $member['recorded_at'] ? date('Y-m-d H:i:s', strtotime($member['recorded_at'])) : null;
    $member['updated_at'] = $member['updated_at'] ? date('Y-m-d H:i:s', strtotime($member['updated_at'])) : null;

    // Fetch allergies
    $allergy_stmt = $conn->prepare("
        SELECT 
            allergen_name,
            type_of_allergy,
            allergy_subtype,
            reaction_type,
            reaction_type_other,
            reaction_severity,
            onset_age,
            notes
        FROM family_history_allergies 
        WHERE family_member_id = ? AND deleted_at IS NULL
    ");
    $allergy_stmt->bind_param("i", $member_id);
    $allergy_stmt->execute();
    $allergy_result = $allergy_stmt->get_result();
    $member['allergies'] = $allergy_result->fetch_all(MYSQLI_ASSOC);
    $allergy_stmt->close();

    // Fetch cancer
    $cancer_stmt = $conn->prepare("
        SELECT 
            cancer_condition,
            cancer_condition_code,
            cancer_diagnosed_age,
            cancer_notes
        FROM family_history_cancer 
        WHERE family_member_id = ? AND deleted_at IS NULL
    ");
    $cancer_stmt->bind_param("i", $member_id);
    $cancer_stmt->execute();
    $cancer_result = $cancer_stmt->get_result();
    $member['cancer'] = $cancer_result->fetch_all(MYSQLI_ASSOC);
    $cancer_stmt->close();

    // Fetch cardiovascular
    $cv_stmt = $conn->prepare("
        SELECT 
            cardiovascular_condition,
            cardiovascular_condition_code,
            cardiovascular_diagnosed_age,
            cardiovascular_notes
        FROM family_history_cardiovascular 
        WHERE family_member_id = ? AND deleted_at IS NULL
    ");
    $cv_stmt->bind_param("i", $member_id);
    $cv_stmt->execute();
    $cv_result = $cv_stmt->get_result();
    $member['cardiovascular'] = $cv_result->fetch_all(MYSQLI_ASSOC);
    $cv_stmt->close();

    // Fetch genetics
    $gen_stmt = $conn->prepare("
        SELECT 
            genetic_condition,
            genetic_condition_code,
            genetic_condition_type_id,
            other_condition_type_name,
            genetic_diagnosed_age,
            genetic_notes,
            carrier_status
        FROM family_history_genetics 
        WHERE family_member_id = ? AND deleted_at IS NULL
    ");
    $gen_stmt->bind_param("i", $member_id);
    $gen_stmt->execute();
    $gen_result = $gen_stmt->get_result();
    $member['genetics'] = $gen_result->fetch_all(MYSQLI_ASSOC);
    $gen_stmt->close();

    // Fetch medical conditions
    $med_stmt = $conn->prepare("
        SELECT 
            medical_condition,
            medical_condition_code,
            medical_diagnosed_age,
            medical_notes
        FROM family_history_medical_conditions 
        WHERE family_member_id = ? AND deleted_at IS NULL
    ");
    $med_stmt->bind_param("i", $member_id);
    $med_stmt->execute();
    $med_result = $med_stmt->get_result();
    $member['medicalConditions'] = $med_result->fetch_all(MYSQLI_ASSOC);
    $med_stmt->close();

    // Fetch psychiatric
    $psy_stmt = $conn->prepare("
        SELECT 
            psychiatric_condition,
            psychiatric_condition_code,
            psychiatric_diagnosed_age,
            psychiatric_notes
        FROM family_history_psychiatric_conditions 
        WHERE family_member_id = ? AND deleted_at IS NULL
    ");
    $psy_stmt->bind_param("i", $member_id);
    $psy_stmt->execute();
    $psy_result = $psy_stmt->get_result();
    $member['psychiatric'] = $psy_result->fetch_all(MYSQLI_ASSOC);
    $psy_stmt->close();

    // Fetch social
    $soc_stmt = $conn->prepare("
        SELECT 
            social_risk_factor,
            severity,
            frequency,
            quantity,
            quantity_unit,
            social_diagnosed_age,
            social_notes
        FROM family_history_social 
        WHERE family_member_id = ? AND deleted_at IS NULL
    ");
    $soc_stmt->bind_param("i", $member_id);
    $soc_stmt->execute();
    $soc_result = $soc_stmt->get_result();
    $member['social'] = $soc_result->fetch_all(MYSQLI_ASSOC);
    $soc_stmt->close();

    $family[] = $member;
}

$stmt->close();

(new Response)->json([
    'patientId' => $patientId,
    'region' => $region,
    'totalFamilyMembers' => count($family),
    'familyHistory' => $family,
    'retrievedAt' => date('c')
]);
?>