<?php
// File name: allergies.php
// File location: /home/apimedi1/public_html/api/v1/routes/medical/allergies.php
require_once __DIR__ . '/../../middleware/jwt.php';
try {
    $userId = requireJWT();
    $patientId = (int)($_GET['patient_id'] ?? ($uri_parts[2] ?? 0));
   
    if ($patientId <= 0) {
        (new Response)->json(['error' => 'Invalid patient ID'], 400);
    }
   
    // Get patient's region and connect to correct medical DB
    $region = getPatientRegion($patientId);
    $conn = getDatabaseConnection('medical_data', $region);
   
    // Check connection
    if (!$conn || $conn->connect_error) {
        throw new Exception("Database connection failed: " . ($conn ? $conn->connect_error : 'No connection'));
    }
   
    $stmt = $conn->prepare("
        SELECT
            id,
            allergen_name,
            type_of_allergy,
            allergy_type_other,
            allergy_subtype,
            reaction_type,
            reaction_type_other,
            allergy_test_type,
            allergy_test_type_other,
            date_of_onset,
            last_occurrence,
            reaction_severity,
            symptoms,
            symptoms_other,
            reaction_duration,
            frequency_of_reactions,
            trigger_factors,
            treatment_management,
            current_management,
            management_other,
            prescribed_medication_for_reaction,
            emergency_protocols,
            emergency_plan,
            emergency_plan_details,
            epinephrine_carried,
            additional_notes,
            validation_status,
            validated_by_user_id,
            validated_by_role_type,
            validated_at,
            validation_reason,
            flagged_as_error,
            error_notes,
            recorded_by_user_id,
            recorded_by_role_type,
            recorded_at,
            updated_by_user_id,
            updated_by_role_type,
            updated_at,
            update_reason
        FROM allergy_history
        WHERE patient_id = ?
          AND deleted_at IS NULL
        ORDER BY date_of_onset DESC, id DESC
    ");
   
    if (!$stmt) {
        throw new Exception("Prepare failed: " . $conn->error);
    }
   
    $stmt->bind_param("i", $patientId);
    $stmt->execute();
    $result = $stmt->get_result();
    $allergies = [];
   
    while ($row = $result->fetch_assoc()) {
        // Convert SET fields to arrays
        $row['symptoms'] = !empty($row['symptoms']) ? explode(',', $row['symptoms']) : [];
        $row['current_management'] = !empty($row['current_management']) ? explode(',', $row['current_management']) : [];
       
        // Format dates
        $row['date_of_onset'] = $row['date_of_onset'] ? date('Y-m-d', strtotime($row['date_of_onset'])) : null;
        $row['last_occurrence'] = $row['last_occurrence'] ? date('Y-m-d', strtotime($row['last_occurrence'])) : null;
        $row['validated_at'] = $row['validated_at'] ? date('Y-m-d H:i:s', strtotime($row['validated_at'])) : null;
        $row['recorded_at'] = $row['recorded_at'] ? date('Y-m-d H:i:s', strtotime($row['recorded_at'])) : null;
        $row['updated_at'] = $row['updated_at'] ? date('Y-m-d H:i:s', strtotime($row['updated_at'])) : null;
       
        $allergies[] = $row;
    }
   
    $stmt->close();
   
    (new Response)->json([
        'patientId' => $patientId,
        'region' => $region,
        'totalAllergies' => count($allergies),
        'allergies' => $allergies,
        'retrievedAt' => date('c')
    ]);
   
} catch (Exception $e) {
    logError($e->getMessage(), 'API Error', __FILE__, __LINE__);
    (new Response)->json([
        'success' => false,
        'error' => 'Service temporarily unavailable. Please try again later.',
        'code' => 'DB_ERROR'
    ], 503);
} finally {
    // Ensure statement and connection resources are cleaned up if needed
    if (isset($stmt)) {
        $stmt->close();
    }
}