<?php
// File name: device-tokens.php
// File location: /home/apimedi1/public_html/api/v1/routes/device-tokens.php
require_once __DIR__ . '/../middleware/jwt.php';
// ------------------------------------------------------------------
// Authentication
// ------------------------------------------------------------------
$header = $_SERVER['HTTP_AUTHORIZATION'] ?? '';
if (!preg_match('/Bearer\s+(\S+)/', $header, $m)) {
    (new Response)->json(['error' => 'Missing token'], 401);
}
$authPayload = validateDeviceOrNormalToken($m[1]);
if (!$authPayload) {
    (new Response)->json(['error' => 'Invalid or revoked token'], 401);
}
$GLOBALS['auth_user'] = $authPayload;
$currentUserId = $authPayload['userId'];
$currentRole = $authPayload['roleType'] ?? 'Patient';

// Update api_requests with correct user_id after successful authentication
$conn_api = getConnAPI();
if ($conn_api && defined('REQUEST_ID')) {
    $request_id = REQUEST_ID;
    $update_stmt = $conn_api->prepare("UPDATE api_requests SET user_id = ? WHERE request_id = ?");
    if ($update_stmt) {
        $update_stmt->bind_param("is", $currentUserId, $request_id);
        $update_stmt->execute();
        $update_stmt->close();
    }
}

$conn = getConnAPI();
$input = getJsonInput();
// ------------------------------------------------------------------
// POST – Create new device token
// ------------------------------------------------------------------
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $deviceName = trim($input['device_name'] ?? '');
    $deviceModel = trim($input['device_model'] ?? '');
    $deviceOs = trim($input['device_os'] ?? '');
    $appName = trim($input['app_name'] ?? 'Meditag App');
    $appVersion = trim($input['app_version'] ?? '');
    $appBuild = trim($input['app_build'] ?? '');
    $roleType = strtoupper(trim($input['role_type'] ?? $currentRole));
    $partnerId = !empty($input['partner_id']) ? (int)$input['partner_id'] : null;
    if (empty($deviceName) || !in_array($roleType, ['PATIENT','PROVIDER','PARTNER','DEVELOPER','STAFF'])) {
        (new Response)->json(['error' => 'device_name and valid role_type required'], 400);
    }
    // Live region
    $mainConn = getConnMain();
    $stmt = $mainConn->prepare("SELECT datacenter_region_code FROM user_datacenter_regions WHERE user_id = ? LIMIT 1");
    $regionCode = 'NA';
    if ($stmt) {
        $stmt->bind_param("i", $currentUserId);
        $stmt->execute();
        $row = $stmt->get_result()->fetch_assoc();
        $regionCode = $row['datacenter_region_code'] ?? 'NA';
    }
    $defaultScopes = json_encode(["profile","medical.read","medical.write","wallet.read"]);
    $token = bin2hex(random_bytes(128));
    $stmt = $conn->prepare("
        INSERT INTO user_device_tokens
        (user_id, role_type, partner_id, device_name, device_model, device_os,
         app_name, app_version, app_build, token, scopes,
         recorded_by_user_id, recorded_by_role_type)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
    ");
    if (!$stmt) {
        (new Response)->json(['error' => 'Database error'], 500);
    }
    $stmt->bind_param("isissssssssis",
        $currentUserId, $roleType, $partnerId, $deviceName, $deviceModel, $deviceOs,
        $appName, $appVersion, $appBuild, $token, $defaultScopes, $currentUserId, $currentRole
    );
    $stmt->execute();
    (new Response)->json([
        'success' => true,
        'message' => 'Device token created successfully',
        'device_token' => $token,
        'region_code' => $regionCode,
        'expires' => 'Never (revoke manually from app settings)',
        'hint' => 'Use this token forever: Authorization: Bearer ' . substr($token, 0, 30) . '...'
    ], 201);
}
// ------------------------------------------------------------------
// GET – List all my devices
// ------------------------------------------------------------------
if ($_SERVER['REQUEST_METHOD'] === 'GET') {
    // Update api_requests with correct user_id (for authenticated list view)
    $conn_api = getConnAPI();
    if ($conn_api && defined('REQUEST_ID')) {
        $request_id = REQUEST_ID;
        $update_stmt = $conn_api->prepare("UPDATE api_requests SET user_id = ? WHERE request_id = ?");
        if ($update_stmt) {
            $update_stmt->bind_param("is", $currentUserId, $request_id);
            $update_stmt->execute();
            $update_stmt->close();
        }
    }

    $query = "
        SELECT
            id,
            device_name,
            app_name,
            app_version,
            role_type,
            COALESCE(partner_id, 0) AS partner_id,
            recorded_at,
            is_revoked,
            last_used_at
        FROM user_device_tokens
        WHERE user_id = ? AND (deleted_at IS NULL)
        ORDER BY recorded_at DESC
    ";
    $stmt = $conn->prepare($query);
    if (!$stmt) {
        (new Response)->json(['success' => false, 'error' => 'Failed to list devices'], 500);
    }
    $stmt->bind_param("i", $currentUserId);
    $stmt->execute();
    $result = $stmt->get_result();
    $devices = $result->fetch_all(MYSQLI_ASSOC);
    (new Response)->json([
        'success' => true,
        'total' => count($devices),
        'devices' => $devices
    ]);
}
// ------------------------------------------------------------------
// DELETE – Revoke a device token (supports URL param OR body)
// ------------------------------------------------------------------
if ($_SERVER['REQUEST_METHOD'] === 'DELETE') {
    // Update api_requests with correct user_id (for authenticated revoke)
    $conn_api = getConnAPI();
    if ($conn_api && defined('REQUEST_ID')) {
        $request_id = REQUEST_ID;
        $update_stmt = $conn_api->prepare("UPDATE api_requests SET user_id = ? WHERE request_id = ?");
        if ($update_stmt) {
            $update_stmt->bind_param("is", $currentUserId, $request_id);
            $update_stmt->execute();
            $update_stmt->close();
        }
    }

    // Prefer URL param from router, fallback to body
    $deviceId = !empty($_GET['device_id']) ? (int)$_GET['device_id'] : (int)($input['device_id'] ?? 0);
    if ($deviceId <= 0) {
        (new Response)->json(['error' => 'device_id required (URL or body)'], 400);
    }
    $stmt = $conn->prepare("
        UPDATE user_device_tokens
        SET is_revoked = 1,
            revoked_at = NOW(),
            revoked_by_user_id = ?,
            revoked_by_role_type = ?
        WHERE id = ? AND user_id = ?
    ");
    if (!$stmt) {
        (new Response)->json(['error' => 'Revoke failed'], 500);
    }
    $stmt->bind_param("isii", $currentUserId, $currentRole, $deviceId, $currentUserId);
    $stmt->execute();
    (new Response)->json(['success' => true, 'message' => 'Device token revoked successfully']);
}
?>