<?php
// File name: register.php
// File location: /home/apimedi1/public_html/api/v1/routes/auth/register.php

require_once __DIR__ . '/../../utils/request.php';
require_once __DIR__ . '/../../handlers/Response.php';
require_once __DIR__ . '/../../../../secure/database_router_v2.php';
require_once __DIR__ . '/../../utils/jwt.php'; // For generateJWT
require_once __DIR__ . '/../../../../vendor/autoload.php'; // For PHPMailer
use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;
require_once __DIR__ . '/../../functions/notifications/sms_bulksms.php'; // For SMS

$response = new Response();
$input = getJsonInput();

// Required fields validation
$required = ['first_name', 'last_name', 'email', 'password', 'mobile_telephone', 'region_code', 'date_of_birth'];
foreach ($required as $field) {
    if (!isset($input[$field]) || empty(trim($input[$field]))) {
        $response->json(['success' => false, 'error' => ucfirst($field) . ' is required'], 400);
    }
}

// Sanitize and validate inputs
$first_name = trim($input['first_name']);
$middle_name = trim($input['middle_name'] ?? '');
$last_name = trim($input['last_name']);
$email = filter_var(trim($input['email']), FILTER_VALIDATE_EMAIL);
if (!$email) {
    $response->json(['success' => false, 'error' => 'Invalid email format'], 400);
}
$password = trim($input['password']);
if (strlen($password) < 8) {
    $response->json(['success' => false, 'error' => 'Password must be at least 8 characters'], 400);
}
$mobile_telephone = trim($input['mobile_telephone']);
$region_code = strtoupper(trim($input['region_code']));
$valid_regions = ['US', 'NA', 'EU', 'ZA'];
if (!in_array($region_code, $valid_regions)) {
    $response->json(['success' => false, 'error' => 'Invalid region code'], 400);
}
$date_of_birth = trim($input['date_of_birth']);
$dob_date = DateTime::createFromFormat('Y-m-d', $date_of_birth);
if (!$dob_date || $dob_date->format('Y-m-d') !== $date_of_birth || $dob_date > new DateTime()) {
    $response->json(['success' => false, 'error' => 'Invalid or future date of birth'], 400);
}
$gender = trim($input['gender'] ?? '');
$valid_genders = ['Male', 'Female', 'Other', 'Prefer not to say'];
if ($gender && !in_array($gender, $valid_genders)) {
    $response->json(['success' => false, 'error' => 'Invalid gender'], 400);
}
$identifications = $input['identifications'] ?? [];
$addresses = $input['addresses'] ?? [];

// Currency map
$currency_map = ['US' => 'USD', 'NA' => 'NAD', 'EU' => 'EUR', 'ZA' => 'ZAR'];
$base_currency = $currency_map[$region_code];

// Timezone (default 'UTC' if not provided)
$timezone = trim($input['timezone'] ?? 'UTC');

// Hash password
$hashed_password = password_hash($password, PASSWORD_DEFAULT);

// Connect to global and regional
$conn_global = getDatabaseConnection('main', 'GBL');
$conn_regional = getDatabaseConnection('main', $region_code);

if (!$conn_global || !$conn_regional) {
    $response->json(['success' => false, 'error' => 'Database connection failed'], 500);
}

// Check duplicate email in global
$stmt = $conn_global->prepare("SELECT id FROM users WHERE email = ?");
$stmt->bind_param("s", $email);
$stmt->execute();
if ($stmt->get_result()->num_rows > 0) {
    $response->json(['success' => false, 'error' => 'Email already exists'], 409);
}
$stmt->close();

// Begin transactions
$conn_global->begin_transaction();
$conn_regional->begin_transaction();

try {
    // Insert into global users
    $stmt_global = $conn_global->prepare("
        INSERT INTO users (first_name, middle_name, last_name, email, password, mobile_telephone, date_of_birth, gender, base_currency, timezone, created_at)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())
    ");
    $stmt_global->bind_param("ssssssssss", $first_name, $middle_name, $last_name, $email, $hashed_password, $mobile_telephone, $date_of_birth, $gender, $base_currency, $timezone);
    $stmt_global->execute();
    $new_user_id = $conn_global->insert_id;
    $stmt_global->close();

    // Insert into regional users
    $stmt_regional_user = $conn_regional->prepare("
        INSERT INTO users (id, first_name, middle_name, last_name, email, password, mobile_telephone, date_of_birth, gender, base_currency, timezone, created_at)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())
    ");
    $stmt_regional_user->bind_param("issssssssss", $new_user_id, $first_name, $middle_name, $last_name, $email, $hashed_password, $mobile_telephone, $date_of_birth, $gender, $base_currency, $timezone);
    $stmt_regional_user->execute();
    $stmt_regional_user->close();

    // Insert identifications if provided (regional)
    foreach ($identifications as $id_entry) {
        $id_type = trim($id_entry['id_type'] ?? '');
        $id_value = trim($id_entry['id_value'] ?? '');
        $issuing_country = trim($id_entry['issuing_country'] ?? $region_code);
        $issuing_authority = trim($id_entry['issuing_authority'] ?? '');
        $issued_date = $id_entry['issued_date'] ?? null;
        $expiry_date = $id_entry['expiry_date'] ?? null;

        if ($id_type && $id_value) {
            $stmt_id = $conn_regional->prepare("
                INSERT INTO user_identifications (user_id, id_type, id_value, issuing_country, issuing_authority, issued_date, expiry_date, status, validation_status, system_created_at, recorded_by_user_id, recorded_by_role_type, recorded_at)
                VALUES (?, ?, ?, ?, ?, ?, ?, 'active', 'Pending Validation', NOW(), ?, 'System', NOW())
            ");
            $stmt_id->bind_param("issssssi", $new_user_id, $id_type, $id_value, $issuing_country, $issuing_authority, $issued_date, $expiry_date, $new_user_id);
            $stmt_id->execute();
            $stmt_id->close();
        }
    }

    // Insert addresses if provided (regional, using upsert)
    foreach ($addresses as $addr) {
        $address_type = trim($addr['address_type'] ?? '');
        $line1 = trim($addr['address_line_1'] ?? '');
        $line2 = trim($addr['address_line_2'] ?? '');
        $city = trim($addr['city'] ?? '');
        $region = trim($addr['region'] ?? '');
        $country = trim($addr['country'] ?? '');
        $zip = trim($addr['zip'] ?? '');

        if ($address_type && $line1 && $city && $region && $country && $zip) {
            // Use upsertAddress function (assume defined in utils or here)
            upsertAddress($conn_regional, $new_user_id, 'Patient', $address_type, $line1, $line2, $city, $region, $country, $zip);
        }
    }

    // Insert into user_datacenter_regions (global)
    $user_role_type = 'Patient'; // Default, as web starts with Patient
    $stmt_region = $conn_global->prepare("
        INSERT INTO user_datacenter_regions (user_id, user_role_type, datacenter_region_code, region_selected, created_at)
        VALUES (?, ?, ?, ?, NOW())
    ");
    $stmt_region->bind_param("isss", $new_user_id, $user_role_type, $region_code, $region_code);
    $stmt_region->execute();
    $stmt_region->close();

    // Send welcome email
    // Fetch settings and template (adapt from web)
    $email_settings = $conn_global->query("SELECT smtp_host, smtp_port, smtp_username, smtp_password, smtp_encryption FROM email_settings LIMIT 1")->fetch_assoc();
    $template = $conn_global->query("SELECT subject, from_email, from_name, body_html FROM email_templates WHERE template_name = 'New User Registration' LIMIT 1")->fetch_assoc();
    $global_template = $conn_global->query("SELECT header_html, footer_html FROM email_templates_global LIMIT 1")->fetch_assoc();

    $mail = new PHPMailer(true);
    $mail->isSMTP();
    $mail->Host = $email_settings['smtp_host'];
    $mail->SMTPAuth = true;
    $mail->Username = $email_settings['smtp_username'];
    $mail->Password = $email_settings['smtp_password'];
    $mail->SMTPSecure = $email_settings['smtp_encryption'];
    $mail->Port = $email_settings['smtp_port'];
    $mail->setFrom($template['from_email'], $template['from_name']);
    $mail->addAddress($email);
    $mail->isHTML(true);
    $mail->Subject = $template['subject'];
    $mail->Body = $global_template['header_html'] . $template['body_html'] . $global_template['footer_html']; // Customize as needed
    $mail->send();

    // Send welcome SMS if mobile provided
    if ($mobile_telephone) {
        $sms_message = "Welcome to Meditag, $first_name! Your account is ready. Username: $email. Please check your email for login instructions. - Meditag Team";
        $providers = $conn_global->query("SELECT * FROM channel_api_providers WHERE channel = 'sms' AND status = 'active' ORDER BY priority_order ASC")->fetch_all(MYSQLI_ASSOC);

        $sendSuccess = false;
        foreach ($providers as $provider) {
            $providerName = $provider['provider_name'];
            if ($providerName === 'BulkSMS') {
                $formatted_number = ltrim($mobile_telephone, '+');
                $url = 'https://api.bulksms.com/v1/messages?auto-unicode=true&longMessageMaxParts=30';
                $messages = [['to' => $formatted_number, 'body' => $sms_message]];
                $result = send_bulksms_message(json_encode($messages), $url, $provider['bulksms_username'], $provider['bulksms_password']);
            } elseif ($providerName === 'Twilio') {
                // Twilio implementation as in web code
                // ...
            }

            $result_status = (isset($result['http_status']) && in_array($result['http_status'], [200, 201])) ? 'sent' : 'failed';
            if ($result_status === 'sent') {
                $sendSuccess = true;
                $sms_id = json_decode($result['server_response'], true)[0]['id'] ?? null;
                log_sms(strtolower($providerName), $sms_id, 'outbound-api', $new_user_id, $mobile_telephone, 'Meditag', $sms_message, 'sent', null, null, $result['http_status'], $result['server_response']);
                break;
            } else {
                log_sms(strtolower($providerName), null, 'outbound-api', $new_user_id, $mobile_telephone, 'Meditag', $sms_message, 'failed', null, null, $result['http_status'], $result['error'] ?? $result['server_response']);
            }
        }
        if (!$sendSuccess) {
            // Log failure but continue
        }
    }

    // Generate JWT
    $jwt = generateJWT($new_user_id, $region_code);

    $conn_global->commit();
    $conn_regional->commit();

    $response->json(['success' => true, 'data' => ['message' => 'Account created successfully', 'user_id' => $new_user_id, 'jwt' => $jwt, 'region_code' => $region_code]], 201);
} catch (Exception $e) {
    $conn_global->rollback();
    $conn_regional->rollback();
    $response->json(['success' => false, 'error' => $e->getMessage()], 500);
}

// Helper function for upsertAddress (from previous code)
function upsertAddress($conn, $userId, $roleType, $addrType, $line1, $line2, $city, $reg, $country, $zip) {
    // Implementation as in guide
    // ...
}