<?php
// File name: wallets.php
// File location: /home/apimedi1/public_html/api/v1/routes/wallet/wallets.php

require_once __DIR__ . '/../../middleware/jwt.php';

$userId = requireJWT();

$method = $_SERVER['REQUEST_METHOD'];
$route = implode('/', array_slice($uri_parts, 0, 2));  // e.g., wallet/balance

// Extract account_number from GET (set by index.php)
$account_number = $_GET['account_number'] ?? '';
if (empty($account_number)) {
    (new Response)->json(['error' => 'Invalid account number'], 400);
}

// Get region
$region = getPatientRegion($userId);
$conn_regional = getDatabaseConnection('wallet', $region);
$conn_global = getConnWallet();
$conn_main = getConnMain();

if ($route === 'wallet/balance') {
    if ($method !== 'GET') {
        (new Response)->json(['error' => 'Method Not Allowed'], 405);
    }

    $stmt = $conn_global->prepare("SELECT account_number, balance, currency, patient_id, provider_id, partner_id FROM wallets WHERE account_number = ? LIMIT 1");
    $stmt->bind_param("s", $account_number);
    $stmt->execute();
    $result = $stmt->get_result();
    $wallet = $result->fetch_assoc();
    $stmt->close();

    if (!$wallet) {
        (new Response)->json(['error' => 'Wallet not found'], 404);
    }

    // Determine type and entity_id
    if ($wallet['patient_id']) {
        $type = 'patient';
        $entity_id = $wallet['patient_id'];
    } elseif ($wallet['provider_id']) {
        $type = 'provider';
        $entity_id = $wallet['provider_id'];
    } elseif ($wallet['partner_id']) {
        $type = 'partner';
        $entity_id = $wallet['partner_id'];
    } else {
        (new Response)->json(['error' => 'Invalid wallet linkage'], 500);
    }

    // Ownership check: fetch linked user_id from core table
    $stmt_owner = $conn_main->prepare("SELECT user_id FROM {$type}s WHERE id = ? LIMIT 1");
    $stmt_owner->bind_param("i", $entity_id);
    $stmt_owner->execute();
    $result_owner = $stmt_owner->get_result();
    $owner = $result_owner->fetch_assoc();
    $stmt_owner->close();

    if (!$owner || $owner['user_id'] != $userId) {
        (new Response)->json(['error' => 'Unauthorized access to wallet'], 403);
    }

    (new Response)->json([
        'type' => $type,
        'account_number' => $wallet['account_number'],
        'region' => $region,
        'balance' => (float)$wallet['balance'],
        'currency' => $wallet['currency'],
        'retrievedAt' => date('c')
    ]);
} elseif ($route === 'wallet/transactions') {
    if ($method !== 'GET') {
        (new Response)->json(['error' => 'Method Not Allowed'], 405);
    }

    // First, check if wallet exists and owned (global check)
    $stmt_check_global = $conn_global->prepare("SELECT patient_id, provider_id, partner_id FROM wallets WHERE account_number = ? LIMIT 1");
    $stmt_check_global->bind_param("s", $account_number);
    $stmt_check_global->execute();
    $result_check_global = $stmt_check_global->get_result();
    $wallet = $result_check_global->fetch_assoc();
    $stmt_check_global->close();

    if (!$wallet) {
        (new Response)->json(['error' => 'Wallet not found'], 404);
    }

    // Determine type and entity_id
    if ($wallet['patient_id']) {
        $type = 'patient';
        $entity_id = $wallet['patient_id'];
    } elseif ($wallet['provider_id']) {
        $type = 'provider';
        $entity_id = $wallet['provider_id'];
    } elseif ($wallet['partner_id']) {
        $type = 'partner';
        $entity_id = $wallet['partner_id'];
    } else {
        (new Response)->json(['error' => 'Invalid wallet linkage'], 500);
    }

    // Ownership check: fetch linked user_id from core table
    $stmt_owner = $conn_main->prepare("SELECT user_id FROM {$type}s WHERE id = ? LIMIT 1");
    $stmt_owner->bind_param("i", $entity_id);
    $stmt_owner->execute();
    $result_owner = $stmt_owner->get_result();
    $owner = $result_owner->fetch_assoc();
    $stmt_owner->close();

    if (!$owner || $owner['user_id'] != $userId) {
        (new Response)->json(['error' => 'Unauthorized access to wallet'], 403);
    }

    // Check regional account
    $stmt_check = $conn_regional->prepare("SELECT id FROM user_bank_accounts WHERE account_number = ? LIMIT 1");
    $stmt_check->bind_param("s", $account_number);
    $stmt_check->execute();
    $result_check = $stmt_check->get_result();
    if ($result_check->num_rows === 0) {
        (new Response)->json(['error' => 'Account not found'], 404);
    }
    $stmt_check->close();

    $stmt = $conn_regional->prepare("SELECT id, transaction_type, transaction_category, amount, currency, transaction_date, description, status FROM transactions WHERE account_number = ? ORDER BY transaction_date DESC LIMIT 50");
    $stmt->bind_param("s", $account_number);
    $stmt->execute();
    $result = $stmt->get_result();

    $transactions = [];
    while ($row = $result->fetch_assoc()) {
        $row['transaction_date'] = date('Y-m-d H:i:s', strtotime($row['transaction_date']));
        $transactions[] = $row;
    }
    $stmt->close();

    (new Response)->json([
        'type' => $type,
        'account_number' => $account_number,
        'region' => $region,
        'totalTransactions' => count($transactions),
        'transactions' => $transactions,
        'retrievedAt' => date('c')
    ]);
} elseif ($route === 'wallet/add-funds') {
    if ($method !== 'POST') {
        (new Response)->json(['error' => 'Method Not Allowed'], 405);
    }

    $data = getJsonInput();
    $amount = (float)($data['amount'] ?? 0);
    $currency = $data['currency'] ?? 'NAD';
    $description = $data['description'] ?? 'Funds added';

    if ($amount <= 0) {
        (new Response)->json(['error' => 'Invalid amount'], 400);
    }

    // Fetch wallet to check existence, currency, and ownership
    $stmt_check = $conn_global->prepare("SELECT account_number, currency, patient_id, provider_id, partner_id FROM wallets WHERE account_number = ? LIMIT 1");
    $stmt_check->bind_param("s", $account_number);
    $stmt_check->execute();
    $result_check = $stmt_check->get_result();
    $wallet = $result_check->fetch_assoc();
    $stmt_check->close();

    if (!$wallet) {
        (new Response)->json(['error' => 'Wallet not found'], 404);
    }

    if ($currency !== $wallet['currency']) {
        (new Response)->json(['error' => 'Currency mismatch'], 400);
    }

    // Determine type and entity_id
    if ($wallet['patient_id']) {
        $type = 'patient';
        $entity_id = $wallet['patient_id'];
    } elseif ($wallet['provider_id']) {
        $type = 'provider';
        $entity_id = $wallet['provider_id'];
    } elseif ($wallet['partner_id']) {
        $type = 'partner';
        $entity_id = $wallet['partner_id'];
    } else {
        (new Response)->json(['error' => 'Invalid wallet linkage'], 500);
    }

    // Ownership check: fetch linked user_id from core table
    $stmt_owner = $conn_main->prepare("SELECT user_id FROM {$type}s WHERE id = ? LIMIT 1");
    $stmt_owner->bind_param("i", $entity_id);
    $stmt_owner->execute();
    $result_owner = $stmt_owner->get_result();
    $owner = $result_owner->fetch_assoc();
    $stmt_owner->close();

    if (!$owner || $owner['user_id'] != $userId) {
        (new Response)->json(['error' => 'Unauthorized access to wallet'], 403);
    }

    // Deposit-specific fields (external top-up = no payer, payee = wallet owner)
    $payer_type_db = null;
    $payer_id_db = null;
    $payee_type_db = strtolower($type); // 'patient' | 'provider' | 'partner'
    $payee_id_db = $entity_id;
    $created_by_role_type = ucfirst($type); // 'Patient' | 'Provider' | 'Partner'

    // Update global balance
    $stmt_global = $conn_global->prepare("UPDATE wallets SET balance = balance + ? WHERE account_number = ?");
    $stmt_global->bind_param("ds", $amount, $account_number);
    if (!$stmt_global->execute()) {
        logError("Global wallet balance update failed for account_number: $account_number - " . $stmt_global->error, 'Wallet', __FILE__, __LINE__);
        (new Response)->json(['error' => 'Failed to update balance'], 500);
    }
    $stmt_global->close();

    // Insert regional transaction with category 'Deposit'
    $stmt_regional = $conn_regional->prepare("
        INSERT INTO transactions 
        (account_number, transaction_type, transaction_category, amount, currency, 
         payer_type, payer_id, payee_type, payee_id, 
         created_by_user_id, created_by_role_type, description, status, created_at) 
        VALUES 
        (?, 'Credit', 'Deposit', ?, ?, 
         ?, ?, ?, ?, 
         ?, ?, ?, 'Pending', NOW())
    ");
    $stmt_regional->bind_param(
        "sdss s i i s s",
        $account_number,
        $amount,
        $currency,
        $payer_type_db,
        $payer_id_db,
        $payee_type_db,
        $payee_id_db,
        $userId,
        $created_by_role_type,
        $description
    );
    if (!$stmt_regional->execute()) {
        logError("Regional transaction insert failed for account_number: $account_number - " . $stmt_regional->error, 'Wallet', __FILE__, __LINE__);
        // Rollback global update if needed (optional - for strict consistency)
        $conn_global->query("UPDATE wallets SET balance = balance - $amount WHERE account_number = '$account_number'");
        (new Response)->json(['error' => 'Failed to record transaction'], 500);
    }
    $stmt_regional->close();

    (new Response)->json([
        'type' => $type,
        'account_number' => $account_number,
        'region' => $region,
        'amountAdded' => $amount,
        'currency' => $currency,
        'message' => 'Funds added successfully (pending confirmation)',
        'addedAt' => date('c')
    ], 201);
} else {
    (new Response)->json(['error' => 'Invalid endpoint'], 404);
}
?>
``````php
<?php
// File name: wallets.php
// File location: /home/apimedi1/public_html/api/v1/routes/wallet/wallets.php

require_once __DIR__ . '/../../middleware/jwt.php';

$userId = requireJWT();

$method = $_SERVER['REQUEST_METHOD'];
$route = implode('/', array_slice($uri_parts, 0, 2));  // e.g., wallet/balance

// Extract account_number from GET (set by index.php)
$account_number = $_GET['account_number'] ?? '';
if (empty($account_number)) {
    (new Response)->json(['error' => 'Invalid account number'], 400);
}

// Get region
$region = getPatientRegion($userId);
$conn_regional = getDatabaseConnection('wallet', $region);
$conn_global = getConnWallet();
$conn_main = getConnMain();

if ($route === 'wallet/balance') {
    if ($method !== 'GET') {
        (new Response)->json(['error' => 'Method Not Allowed'], 405);
    }

    $stmt = $conn_global->prepare("SELECT account_number, balance, currency, patient_id, provider_id, partner_id FROM wallets WHERE account_number = ? LIMIT 1");
    $stmt->bind_param("s", $account_number);
    $stmt->execute();
    $result = $stmt->get_result();
    $wallet = $result->fetch_assoc();
    $stmt->close();

    if (!$wallet) {
        (new Response)->json(['error' => 'Wallet not found'], 404);
    }

    // Determine type and entity_id
    if ($wallet['patient_id']) {
        $type = 'patient';
        $entity_id = $wallet['patient_id'];
    } elseif ($wallet['provider_id']) {
        $type = 'provider';
        $entity_id = $wallet['provider_id'];
    } elseif ($wallet['partner_id']) {
        $type = 'partner';
        $entity_id = $wallet['partner_id'];
    } else {
        (new Response)->json(['error' => 'Invalid wallet linkage'], 500);
    }

    // Ownership check: fetch linked user_id from core table
    $stmt_owner = $conn_main->prepare("SELECT user_id FROM {$type}s WHERE id = ? LIMIT 1");
    $stmt_owner->bind_param("i", $entity_id);
    $stmt_owner->execute();
    $result_owner = $stmt_owner->get_result();
    $owner = $result_owner->fetch_assoc();
    $stmt_owner->close();

    if (!$owner || $owner['user_id'] != $userId) {
        (new Response)->json(['error' => 'Unauthorized access to wallet'], 403);
    }

    (new Response)->json([
        'type' => $type,
        'account_number' => $wallet['account_number'],
        'region' => $region,
        'balance' => (float)$wallet['balance'],
        'currency' => $wallet['currency'],
        'retrievedAt' => date('c')
    ]);
} elseif ($route === 'wallet/transactions') {
    if ($method !== 'GET') {
        (new Response)->json(['error' => 'Method Not Allowed'], 405);
    }

    // First, check if wallet exists and owned (global check)
    $stmt_check_global = $conn_global->prepare("SELECT patient_id, provider_id, partner_id FROM wallets WHERE account_number = ? LIMIT 1");
    $stmt_check_global->bind_param("s", $account_number);
    $stmt_check_global->execute();
    $result_check_global = $stmt_check_global->get_result();
    $wallet = $result_check_global->fetch_assoc();
    $stmt_check_global->close();

    if (!$wallet) {
        (new Response)->json(['error' => 'Wallet not found'], 404);
    }

    // Determine type and entity_id
    if ($wallet['patient_id']) {
        $type = 'patient';
        $entity_id = $wallet['patient_id'];
    } elseif ($wallet['provider_id']) {
        $type = 'provider';
        $entity_id = $wallet['provider_id'];
    } elseif ($wallet['partner_id']) {
        $type = 'partner';
        $entity_id = $wallet['partner_id'];
    } else {
        (new Response)->json(['error' => 'Invalid wallet linkage'], 500);
    }

    // Ownership check: fetch linked user_id from core table
    $stmt_owner = $conn_main->prepare("SELECT user_id FROM {$type}s WHERE id = ? LIMIT 1");
    $stmt_owner->bind_param("i", $entity_id);
    $stmt_owner->execute();
    $result_owner = $stmt_owner->get_result();
    $owner = $result_owner->fetch_assoc();
    $stmt_owner->close();

    if (!$owner || $owner['user_id'] != $userId) {
        (new Response)->json(['error' => 'Unauthorized access to wallet'], 403);
    }

    // Check regional account
    $stmt_check = $conn_regional->prepare("SELECT id FROM user_bank_accounts WHERE account_number = ? LIMIT 1");
    $stmt_check->bind_param("s", $account_number);
    $stmt_check->execute();
    $result_check = $stmt_check->get_result();
    if ($result_check->num_rows === 0) {
        (new Response)->json(['error' => 'Account not found'], 404);
    }
    $stmt_check->close();

    $stmt = $conn_regional->prepare("SELECT id, transaction_type, transaction_category, amount, currency, transaction_date, description, status FROM transactions WHERE account_number = ? ORDER BY transaction_date DESC LIMIT 50");
    $stmt->bind_param("s", $account_number);
    $stmt->execute();
    $result = $stmt->get_result();

    $transactions = [];
    while ($row = $result->fetch_assoc()) {
        $row['transaction_date'] = date('Y-m-d H:i:s', strtotime($row['transaction_date']));
        $transactions[] = $row;
    }
    $stmt->close();

    (new Response)->json([
        'type' => $type,
        'account_number' => $account_number,
        'region' => $region,
        'totalTransactions' => count($transactions),
        'transactions' => $transactions,
        'retrievedAt' => date('c')
    ]);
} elseif ($route === 'wallet/add-funds') {
    if ($method !== 'POST') {
        (new Response)->json(['error' => 'Method Not Allowed'], 405);
    }

    $data = getJsonInput();
    $amount = (float)($data['amount'] ?? 0);
    $currency = $data['currency'] ?? 'NAD';
    $description = $data['description'] ?? 'Funds added';

    if ($amount <= 0) {
        (new Response)->json(['error' => 'Invalid amount'], 400);
    }

    // Fetch wallet to check existence, currency, and ownership
    $stmt_check = $conn_global->prepare("SELECT account_number, currency, patient_id, provider_id, partner_id FROM wallets WHERE account_number = ? LIMIT 1");
    $stmt_check->bind_param("s", $account_number);
    $stmt_check->execute();
    $result_check = $stmt_check->get_result();
    $wallet = $result_check->fetch_assoc();
    $stmt_check->close();

    if (!$wallet) {
        (new Response)->json(['error' => 'Wallet not found'], 404);
    }

    if ($currency !== $wallet['currency']) {
        (new Response)->json(['error' => 'Currency mismatch'], 400);
    }

    // Determine type and entity_id
    if ($wallet['patient_id']) {
        $type = 'patient';
        $entity_id = $wallet['patient_id'];
    } elseif ($wallet['provider_id']) {
        $type = 'provider';
        $entity_id = $wallet['provider_id'];
    } elseif ($wallet['partner_id']) {
        $type = 'partner';
        $entity_id = $wallet['partner_id'];
    } else {
        (new Response)->json(['error' => 'Invalid wallet linkage'], 500);
    }

    // Ownership check: fetch linked user_id from core table
    $stmt_owner = $conn_main->prepare("SELECT user_id FROM {$type}s WHERE id = ? LIMIT 1");
    $stmt_owner->bind_param("i", $entity_id);
    $stmt_owner->execute();
    $result_owner = $stmt_owner->get_result();
    $owner = $result_owner->fetch_assoc();
    $stmt_owner->close();

    if (!$owner || $owner['user_id'] != $userId) {
        (new Response)->json(['error' => 'Unauthorized access to wallet'], 403);
    }

    // Deposit-specific fields (external top-up = no payer, payee = wallet owner)
    $payer_type_db = null;
    $payer_id_db = null;
    $payee_type_db = strtolower($type); // 'patient' | 'provider' | 'partner'
    $payee_id_db = $entity_id;
    $created_by_role_type = ucfirst($type); // 'Patient' | 'Provider' | 'Partner'

    // Update global balance
    $stmt_global = $conn_global->prepare("UPDATE wallets SET balance = balance + ? WHERE account_number = ?");
    $stmt_global->bind_param("ds", $amount, $account_number);
    if (!$stmt_global->execute()) {
        logError("Global wallet balance update failed for account_number: $account_number - " . $stmt_global->error, 'Wallet', __FILE__, __LINE__);
        (new Response)->json(['error' => 'Failed to update balance'], 500);
    }
    $stmt_global->close();

    // Insert regional transaction with category 'Deposit'
    $stmt_regional = $conn_regional->prepare("
        INSERT INTO transactions 
        (account_number, transaction_type, transaction_category, amount, currency, 
         payer_type, payer_id, payee_type, payee_id, 
         created_by_user_id, created_by_role_type, description, status, created_at) 
        VALUES 
        (?, 'Credit', 'Deposit', ?, ?, 
         ?, ?, ?, ?, 
         ?, ?, ?, 'Pending', NOW())
    ");
    $stmt_regional->bind_param(
        "sdsssiisss",
        $account_number,
        $amount,
        $currency,
        $payer_type_db,
        $payer_id_db,
        $payee_type_db,
        $payee_id_db,
        $userId,
        $created_by_role_type,
        $description
    );
    if (!$stmt_regional->execute()) {
        logError("Regional transaction insert failed for account_number: $account_number - " . $stmt_regional->error, 'Wallet', __FILE__, __LINE__);
        // Rollback global update if needed (optional - for strict consistency)
        $conn_global->query("UPDATE wallets SET balance = balance - $amount WHERE account_number = '$account_number'");
        (new Response)->json(['error' => 'Failed to record transaction'], 500);
    }
    $stmt_regional->close();

    (new Response)->json([
        'type' => $type,
        'account_number' => $account_number,
        'region' => $region,
        'amountAdded' => $amount,
        'currency' => $currency,
        'message' => 'Funds added successfully (pending confirmation)',
        'addedAt' => date('c')
    ], 201);
} else {
    (new Response)->json(['error' => 'Invalid endpoint'], 404);
}
?>