<?php
// File name: balance.php

// File location: /home/apimedi1/public_html/api/v1/routes/wallet/balance.php

require_once __DIR__ . '/../../middleware/jwt.php';
$userId = requireJWT();

$account_number = $_GET['account_number'] ?? '';
if (empty($account_number)) {
    (new Response)->json(['error' => 'Invalid account number'], 400);
}

$region = getPatientRegion($userId);
$conn_regional = getDatabaseConnection('wallet', $region);
$conn_global = getConnWallet();
$conn_main = getConnMain();

$stmt = $conn_global->prepare("SELECT account_number, balance, currency, patient_id, provider_id, partner_id FROM wallets WHERE account_number = ? LIMIT 1");
$stmt->bind_param("s", $account_number);
$stmt->execute();
$result = $stmt->get_result();
$wallet = $result->fetch_assoc();
$stmt->close();

if (!$wallet) {
    (new Response)->json(['error' => 'Wallet not found'], 404);
}

if ($wallet['patient_id']) {
    $type = 'patient'; $entity_id = $wallet['patient_id'];
} elseif ($wallet['provider_id']) {
    $type = 'provider'; $entity_id = $wallet['provider_id'];
} elseif ($wallet['partner_id']) {
    $type = 'partner'; $entity_id = $wallet['partner_id'];
} else {
    (new Response)->json(['error' => 'Invalid wallet linkage'], 500);
}

$stmt_owner = $conn_main->prepare("SELECT user_id FROM {$type}s WHERE id = ? LIMIT 1");
$stmt_owner->bind_param("i", $entity_id);
$stmt_owner->execute();
$owner = $stmt_owner->get_result()->fetch_assoc();
$stmt_owner->close();

if (!$owner || $owner['user_id'] != $userId) {
    (new Response)->json(['error' => 'Unauthorized access to wallet'], 403);
}

// Calculate running balance from completed transactions (override DB balance if needed)
$stmt_running = $conn_regional->prepare("
    SELECT transaction_type, amount 
    FROM transactions 
    WHERE account_number = ? AND status = 'Completed' 
    ORDER BY transaction_date ASC
");
$stmt_running->bind_param("s", $account_number);
$stmt_running->execute();
$all_tx = $stmt_running->get_result()->fetch_all(MYSQLI_ASSOC);
$stmt_running->close();

$running_balance = 0.0;
foreach ($all_tx as $tx) {
    if ($tx['transaction_type'] === 'Credit') {
        $running_balance += (float)$tx['amount'];
    } else {
        $running_balance -= (float)$tx['amount'];
    }
}

// Pending credits (Deposits pending)
$stmt_credits = $conn_regional->prepare("
    SELECT SUM(amount) AS sum_credits, COUNT(*) AS count_credits 
    FROM transactions 
    WHERE account_number = ? AND status = 'Pending' AND transaction_category = 'Deposit'
");
$stmt_credits->bind_param("s", $account_number);
$stmt_credits->execute();
$credits = $stmt_credits->get_result()->fetch_assoc();
$stmt_credits->close();
$pending_credits_amount = (float)($credits['sum_credits'] ?? 0);
$pending_credits_count = (int)($credits['count_credits'] ?? 0);

// Pending debits
$stmt_debits = $conn_regional->prepare("
    SELECT SUM(amount) AS sum_debits, COUNT(*) AS count_debits 
    FROM transactions 
    WHERE account_number = ? AND status = 'Pending' AND transaction_type = 'Debit'
");
$stmt_debits->bind_param("s", $account_number);
$stmt_debits->execute();
$debits = $stmt_debits->get_result()->fetch_assoc();
$stmt_debits->close();
$pending_debits_amount = (float)($debits['sum_debits'] ?? 0);
$pending_debits_count = (int)($debits['count_debits'] ?? 0);

// Reserved funds (Holds)
$stmt_reserved = $conn_regional->prepare("
    SELECT SUM(amount) AS sum_reserved, COUNT(*) AS count_reserved 
    FROM transactions 
    WHERE account_number = ? AND status = 'Hold' AND transaction_type = 'Debit'
");
$stmt_reserved->bind_param("s", $account_number);
$stmt_reserved->execute();
$reserved = $stmt_reserved->get_result()->fetch_assoc();
$stmt_reserved->close();
$reserved_amount = (float)($reserved['sum_reserved'] ?? 0);
$reserved_count = (int)($reserved['count_reserved'] ?? 0);

// Computed balances
$total_balance = $running_balance + $pending_credits_amount - $pending_debits_amount;
$available_balance = $running_balance - $pending_debits_amount - $reserved_amount;

(new Response)->json([
    'type' => $type,
    'account_number' => $wallet['account_number'],
    'region' => $region,
    'runningBalance' => $running_balance,
    'pendingCredits' => [
        'amount' => $pending_credits_amount,
        'count' => $pending_credits_count
    ],
    'pendingDebits' => [
        'amount' => $pending_debits_amount,
        'count' => $pending_debits_count
    ],
    'reservedFunds' => [
        'amount' => $reserved_amount,
        'count' => $reserved_count
    ],
    'totalBalance' => $total_balance,
    'availableBalance' => $available_balance,
    'currency' => $wallet['currency'],
    'retrievedAt' => date('c')
]);
?>