<?php
// File name: vehicles.php
// File location: /home/apimedi1/public_html/api/v1/routes/panic/vehicles.php

require_once __DIR__ . '/../../middleware/jwt.php';
require_once __DIR__ . '/../../utils/request.php';
require_once __DIR__ . '/../../../../../secure/config_v2.php';
require_once __DIR__ . '/../../../../../secure/database_router_v2.php';

$response = new Response();
$input = getJsonInput();
$method = $_SERVER['REQUEST_METHOD'];

$conn_panic = getDatabaseConnection('panic_service', 'GBL');
if (!$conn_panic) {
    $response->json(['error' => 'Database connection failed'], 500);
}

switch ($method) {
    case 'POST':
        // Create new vehicle
        requireJWT(); // Admin only, add middleware/admin.php if needed
        $required = ['license_plate', 'vehicle_type'];
        $missing = array_diff($required, array_keys($input));
        if (!empty($missing)) {
            $response->json(['error' => 'Missing fields: ' . implode(', ', $missing)], 400);
        }

        $license_plate = $input['license_plate'];
        $make = $input['make'] ?? null;
        $model = $input['model'] ?? null;
        $vehicle_type = $input['vehicle_type'];

        $allowed_types = ['ambulance', 'police', 'fire', 'other'];
        if (!in_array($vehicle_type, $allowed_types)) {
            $response->json(['error' => 'Invalid vehicle_type'], 400);
        }

        $stmt = $conn_panic->prepare("
            INSERT INTO vehicles (license_plate, make, model, vehicle_type, created_at)
            VALUES (?, ?, ?, ?, NOW())
        ");
        $stmt->bind_param('ssss', $license_plate, $make, $model, $vehicle_type);
        if ($stmt->execute()) {
            $vehicle_id = $stmt->insert_id;
            $response->json(['success' => true, 'vehicle_id' => $vehicle_id], 201);
        } else {
            $response->json(['success' => false, 'message' => $stmt->error], 500);
        }
        $stmt->close();
        break;

    case 'GET':
        // Get vehicle(s) - by ID or list
        $vehicle_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
        if ($vehicle_id) {
            $stmt = $conn_panic->prepare("SELECT * FROM vehicles WHERE id = ?");
            $stmt->bind_param('i', $vehicle_id);
        } else {
            $stmt = $conn_panic->prepare("SELECT * FROM vehicles ORDER BY created_at DESC LIMIT 50"); // Paginate if needed
        }
        $stmt->execute();
        $result = $stmt->get_result();
        $vehicles = [];
        while ($row = $result->fetch_assoc()) {
            $vehicles[] = $row;
        }
        $stmt->close();
        $response->json($vehicles, 200);
        break;

    case 'PATCH':
        // Update vehicle
        requireJWT();
        $vehicle_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
        if (!$vehicle_id) {
            $response->json(['error' => 'Missing vehicle_id'], 400);
        }

        $updates = [];
        $params = [];
        $types = '';

        if (isset($input['license_plate'])) {
            $updates[] = 'license_plate = ?';
            $types .= 's';
            $params[] = $input['license_plate'];
        }
        if (isset($input['make'])) {
            $updates[] = 'make = ?';
            $types .= 's';
            $params[] = $input['make'];
        }
        if (isset($input['model'])) {
            $updates[] = 'model = ?';
            $types .= 's';
            $params[] = $input['model'];
        }
        if (isset($input['vehicle_type'])) {
            $updates[] = 'vehicle_type = ?';
            $types .= 's';
            $params[] = $input['vehicle_type'];
        }

        if (empty($updates)) {
            $response->json(['error' => 'No fields to update'], 400);
        }

        $sql = "UPDATE vehicles SET " . implode(', ', $updates) . " WHERE id = ?";
        $types .= 'i';
        $params[] = $vehicle_id;

        $stmt = $conn_panic->prepare($sql);
        $stmt->bind_param($types, ...$params);
        if ($stmt->execute()) {
            $response->json(['success' => true, 'message' => 'Vehicle updated'], 200);
        } else {
            $response->json(['success' => false, 'message' => $stmt->error], 500);
        }
        $stmt->close();
        break;

    case 'DELETE':
        // Delete vehicle
        requireJWT();
        $vehicle_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
        if (!$vehicle_id) {
            $response->json(['error' => 'Missing vehicle_id'], 400);
        }

        $stmt = $conn_panic->prepare("DELETE FROM vehicles WHERE id = ?");
        $stmt->bind_param('i', $vehicle_id);
        if ($stmt->execute()) {
            $response->json(['success' => true, 'message' => 'Vehicle deleted'], 200);
        } else {
            $response->json(['success' => false, 'message' => $stmt->error], 500);
        }
        $stmt->close();
        break;

    default:
        $response->json(['error' => 'Method not allowed'], 405);
}