<?php
// File name: events.php
// File location: /home/apimedi1/public_html/api/v1/routes/panic/events.php
ini_set('max_execution_time', 300); // 5 min timeout
ini_set('default_socket_timeout', 60);
require_once __DIR__ . '/../../utils/jwt.php';
require_once __DIR__ . '/../../middleware/jwt.php';
require_once __DIR__ . '/../../utils/region.php';
require_once __DIR__ . '/../../utils/request.php';
require_once __DIR__ . '/../../handlers/Response.php';
require_once __DIR__ . '/../../../../../secure/config_v2.php';
require_once __DIR__ . '/../../../../../secure/database_router_v2.php';
require_once __DIR__ . '/../../functions/notifications/sms_bulksms.php';
function prepareOrThrow($conn, $sql, $label) {
    $stmt = $conn->prepare($sql);
    if (!$stmt) {
        throw new Exception("Prepare failed ($label): (" . $conn->errno . ") " . $conn->error);
    }
    return $stmt;
}
function executeOrThrow($stmt, $label) {
    if (!$stmt->execute()) {
        throw new Exception("Execute failed ($label): (" . $stmt->errno . ") " . $stmt->error);
    }
}
$response = new Response();
$input = getJsonInput();
$method = $_SERVER['REQUEST_METHOD'];
$conn_panic = getDatabaseConnection('panic_service', 'GBL');
if (!$conn_panic) {
    error_log("Panic DB connection failed: " . json_encode(mysqli_connect_error()));
    $response->json(['error' => 'Panic database connection failed'], 500);
}
$result = $conn_panic->query("SELECT DATABASE() as db");
$row = $result ? $result->fetch_assoc() : null;
error_log("panic DB in use: " . ($row['db'] ?? 'unknown'));
switch ($method) {
    case 'POST':
        // Store new panic event and send notifications
        requireJWT(); // Requires JWT authentication
        $required = ['user_id', 'emergency_types', 'latitude', 'longitude'];
        $missing = array_diff($required, array_keys($input));
        if (!empty($missing)) {
            $response->json(['success' => false, 'message' => 'Missing fields: ' . implode(', ', $missing)], 400);
        }
        $user_id = (int)$input['user_id'];
        $patient_id = isset($input['patient_id']) ? (int)$input['patient_id'] : null;
        $emergency_types = (array)$input['emergency_types'];
        $latitude = (float)$input['latitude'];
        $longitude = (float)$input['longitude'];
        // Optional fields with defaults
        $battery_level = isset($input['battery_level']) ? (int)$input['battery_level'] : 0;
        $network_status = isset($input['network_status']) ? $input['network_status'] : '';
        $network_name = isset($input['network_name']) ? $input['network_name'] : '';
        $device_type = isset($input['device_type']) ? $input['device_type'] : '';
        $device_model = isset($input['device_model']) ? $input['device_model'] : '';
        $operating_sys = isset($input['operating_system']) ? $input['operating_system'] : '';
        $platform = isset($input['platform']) ? $input['platform'] : '';
        $user_agent = isset($input['browser_user_agent']) ? $input['browser_user_agent'] : '';
        $app_version = isset($input['app_version']) ? $input['app_version'] : '';
        $loc_accuracy = isset($input['location_accuracy']) ? (float)$input['location_accuracy'] : 0.0;
        $altitude = isset($input['altitude']) ? (float)$input['altitude'] : 0.0;
        $heading = isset($input['heading']) ? (float)$input['heading'] : 0.0;
        $speed = isset($input['speed']) ? (float)$input['speed'] : 0.0;
        $connection_type = isset($input['connection_type']) ? $input['connection_type'] : '';
        $ip_address = $_SERVER['REMOTE_ADDR'] ?? 'unknown'; // Server-side IP
        $conn_panic->begin_transaction();
        try {
            $start_time = microtime(true); // Timing: Start overall
            // Fetch regionCode
            $regionCode = 'NA'; // fallback
            $conn_main = getDatabaseConnection('main', 'GBL');
            if (!$conn_main) {
                throw new Exception("Main DB connection failed");
            }
            $stmt_region = prepareOrThrow($conn_main, "SELECT datacenter_region_code FROM user_datacenter_regions WHERE user_id = ? AND user_role_type = 'Patient' AND deleted_at IS NULL LIMIT 1", 'user_region');
            $stmt_region->bind_param('i', $user_id);
            executeOrThrow($stmt_region, 'user_region');
            $result_region = $stmt_region->get_result();
            if ($row = $result_region->fetch_assoc()) {
                $regionCode = strtoupper($row['datacenter_region_code']);
            }
            $stmt_region->close();
            error_log("Timing: Region fetch took " . (microtime(true) - $start_time) . " seconds");
            $start_time = microtime(true); // Reset for next section
            // Insert into panic_button_events
            $stmt_event = prepareOrThrow($conn_panic, "
                INSERT INTO panic_button_events (user_id, latitude, longitude, ip_address, created_at)
                VALUES (?, ?, ?, ?, NOW())
            ", 'panic_button_events');
            $stmt_event->bind_param('idds', $user_id, $latitude, $longitude, $ip_address);
            executeOrThrow($stmt_event, 'panic_button_events');
            $event_id = $conn_panic->insert_id;
            $stmt_event->close();
            error_log("Timing: Event insert took " . (microtime(true) - $start_time) . " seconds");
            $start_time = microtime(true);
            // Insert into panic_event_types
            $allowed_types = ['flood', 'fire', 'medical', 'security'];
            $stmt_type = prepareOrThrow($conn_panic, "
                INSERT INTO panic_event_types (event_id, emergency_type)
                VALUES (?, ?)
            ", 'panic_event_types');
            foreach ($emergency_types as $type) {
                $type = strtolower(trim($type));
                if (!in_array($type, $allowed_types)) {
                    throw new Exception("Invalid emergency type: $type");
                }
                $stmt_type->bind_param('is', $event_id, $type);
                executeOrThrow($stmt_type, 'panic_event_types');
            }
            $stmt_type->close();
            error_log("Timing: Event types insert took " . (microtime(true) - $start_time) . " seconds");
            $start_time = microtime(true);
            // Insert into patient_locations if patient_id provided
            if ($patient_id) {
                $stmt_patient = prepareOrThrow($conn_panic, "
                    INSERT INTO patient_locations
                    (patient_id, latitude, longitude, battery_level, network_strength, network_name, ip_address,
                    device_type, device_model, operating_system, platform, browser_user_agent, app_version,
                    location_accuracy, altitude, heading, speed, connection_type)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                ", 'patient_locations');
                $stmt_patient->bind_param('iddisssssssssdddds', $patient_id, $latitude, $longitude, $battery_level,
                    $network_status, $network_name, $ip_address, $device_type, $device_model, $operating_sys,
                    $platform, $user_agent, $app_version, $loc_accuracy, $altitude, $heading, $speed, $connection_type);
                executeOrThrow($stmt_patient, 'patient_locations');
                $stmt_patient->close();
            }
            error_log("Timing: Patient location insert took " . (microtime(true) - $start_time) . " seconds");
            $start_time = microtime(true);
            // Generate secure key and tracking link
            $key = bin2hex(random_bytes(16));
            $domain = 'meditag.net'; // Fallback
            $stmtDomain = prepareOrThrow($conn_main, "SELECT domain FROM meditag_entities WHERE is_headquarters = 1 AND deleted_at IS NULL LIMIT 1", 'meditag_entities');
            executeOrThrow($stmtDomain, 'meditag_entities');
            $resDomain = $stmtDomain->get_result();
            $domainRow = $resDomain->fetch_assoc();
            $domain = $domainRow['domain'] ?? $domain;
            $stmtDomain->close();
            $tracking_link = "https://www.$domain/emergency_track.php?event_id=" . $event_id . "&key=" . $key;
            // Update event description with link
            $emergencyTypeStr = implode(', ', $emergency_types);
            $newDescription = "Meditag Alert: Emergency reported: {$emergencyTypeStr}. Track this incident here: {$tracking_link}";
            $stmtUpdate = prepareOrThrow($conn_panic, "UPDATE panic_button_events SET description = ? WHERE id = ?", 'panic_button_events_update');
            $stmtUpdate->bind_param("si", $newDescription, $event_id);
            executeOrThrow($stmtUpdate, 'panic_button_events_update');
            $stmtUpdate->close();
            error_log("Timing: Tracking link generation and update took " . (microtime(true) - $start_time) . " seconds");
            $start_time = microtime(true);
            // Fetch emergency contacts (region-routed)
            $conn_medical = getDatabaseConnection('medical_data', $regionCode);
            if (!$conn_medical) {
                throw new Exception("Medical DB connection failed for region: $regionCode");
            }
            $stmtContacts = prepareOrThrow($conn_medical, "SELECT phone_number FROM emergency_contacts WHERE patient_id = ? AND deleted_at IS NULL AND phone_number IS NOT NULL AND phone_number != ''", 'emergency_contacts');
            $stmtContacts->bind_param('i', $patient_id);
            executeOrThrow($stmtContacts, 'emergency_contacts');
            $resContacts = $stmtContacts->get_result();
            $familyNumbers = [];
            while ($row = $resContacts->fetch_assoc()) {
                $phone = trim($row['phone_number']);
                if (preg_match('/^\+\d{10,15}$/', $phone)) {
                    $familyNumbers[] = $phone;
                }
            }
            $stmtContacts->close();
            // Prevent duplicate sends by making numbers unique
            $familyNumbers = array_unique($familyNumbers);
            if (empty($familyNumbers)) {
                $fallbackNumber = '+26481225';
                if (preg_match('/^\+\d{10,15}$/', $fallbackNumber)) {
                    $familyNumbers[] = $fallbackNumber;
                }
            }
            error_log("Timing: Emergency contacts fetch took " . (microtime(true) - $start_time) . " seconds");
            $start_time = microtime(true);
            // Prepare SMS body
            $smsBody = "Meditag Alert: You are an emergency contact for a Meditag profile. An emergency of type '{$emergencyTypeStr}' has been triggered. To view live updates, please visit: {$tracking_link}";
            // Fetch active SMS providers
            $stmtProviders = prepareOrThrow($conn_main, "SELECT provider_name, bulksms_username, bulksms_password, twilio_account_sid, twilio_auth_token, twilio_phone_number, sender_id, other_params
                             FROM channel_api_providers WHERE channel = 'sms' AND status = 'active' ORDER BY priority_order ASC", 'channel_api_providers');
            executeOrThrow($stmtProviders, 'channel_api_providers');
            $resProviders = $stmtProviders->get_result();
            $providers = [];
            while ($row = $resProviders->fetch_assoc()) {
                $providers[] = $row;
            }
            $stmtProviders->close();
            if (empty($providers)) {
                throw new Exception('No active SMS providers available.');
            }
            error_log("Timing: SMS providers fetch took " . (microtime(true) - $start_time) . " seconds");
            // Insert event updates
            $qr_scan_time = date('Y-m-d H:i:s'); // Fetch from qr_code_scans if scan_id provided, else current
            $update_message1 = "Patient EMI QR Code scanned at " . $qr_scan_time . " UTC.";
            $stmt_update1 = prepareOrThrow($conn_panic, "INSERT INTO event_updates (event_id, update_message, updated_by, `key`) VALUES (?, ?, ?, ?)", 'event_updates1');
            $stmt_update1->bind_param("isis", $event_id, $update_message1, $user_id, $key);
            executeOrThrow($stmt_update1, 'event_updates1');
            $stmt_update1->close();
            $update_message2 = "Emergency reported. SMS notifications sent to emergency contacts with tracking link.";
            $stmt_update2 = prepareOrThrow($conn_panic, "INSERT INTO event_updates (event_id, update_message, updated_by, `key`) VALUES (?, ?, ?, ?)", 'event_updates2');
            $stmt_update2->bind_param("isis", $event_id, $update_message2, $user_id, $key);
            executeOrThrow($stmt_update2, 'event_updates2');
            $stmt_update2->close();
            error_log("Timing: Event updates insert took " . (microtime(true) - $start_time) . " seconds");
            $conn_panic->commit();
            $start_time = microtime(true);
            // Send SMS to each contact with failover
            $smsSent = [];
            $conn_audit = getDatabaseConnection('audit', 'GBL');
            if (!$conn_audit) {
                throw new Exception("Audit DB connection failed");
            }
            foreach ($familyNumbers as $toNumber) {
                $sms_start = microtime(true); // Per-SMS timing
                $sendSuccess = false;
                $successfulProvider = null;
                $successfulResult = null;
                $date_created = date('Y-m-d H:i:s');
                foreach ($providers as $provider) {
                    $result = null;
                    if (strcasecmp($provider['provider_name'], 'BulkSMS') === 0) {
                        $formatted_number = (strpos($toNumber, '+') === 0) ? substr($toNumber, 1) : $toNumber;
                        $url = 'https://api.bulksms.com/v1/messages?auto-unicode=true&longMessageMaxParts=30';
                        $messages = [['to' => $formatted_number, 'body' => $smsBody]];
                        $post_body = json_encode($messages);
                        $result = send_bulksms_message($post_body, $url, $provider['bulksms_username'], $provider['bulksms_password']);
                    } elseif (strcasecmp($provider['provider_name'], 'Twilio') === 0) {
                        if (empty($provider['twilio_account_sid']) || empty($provider['twilio_auth_token']) || empty($provider['twilio_phone_number'])) {
                            continue;
                        }
                        $url = "https://api.twilio.com/2010-04-01/Accounts/{$provider['twilio_account_sid']}/Messages.json";
                        $from_number = (strpos($provider['twilio_phone_number'], '+') === 0) ? $provider['twilio_phone_number'] : '+' . $provider['twilio_phone_number'];
                        $fields = http_build_query(['To' => $toNumber, 'From' => $from_number, 'Body' => $smsBody]);
                        $ch = curl_init();
                        curl_setopt($ch, CURLOPT_URL, $url);
                        curl_setopt($ch, CURLOPT_POST, true);
                        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                        curl_setopt($ch, CURLOPT_POSTFIELDS, $fields);
                        curl_setopt($ch, CURLOPT_USERPWD, "{$provider['twilio_account_sid']}:{$provider['twilio_auth_token']}");
                        curl_setopt($ch, CURLOPT_TIMEOUT, 30); // Increased to 30s total timeout
                        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 15); // Increased to 15s connect timeout
                        $server_response = curl_exec($ch);
                        $http_status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                        $error = curl_error($ch);
                        curl_close($ch);
                        $result = ['server_response' => $server_response, 'http_status' => $http_status, 'error' => $error];
                    } else {
                        continue;
                    }
                    $status = ($result['http_status'] == 201 || $result['http_status'] == 200) ? 'sent' : 'failed';
                    if ($status == 'sent') {
                        $sendSuccess = true;
                        $successfulResult = $result;
                        $successfulProvider = $provider;
                        break;
                    } else {
                        // Log failed attempt
                        log_sms(
                            strtolower($provider['provider_name']),
                            null,
                            'outbound-api',
                            $patient_id,
                            $toNumber,
                            'Meditag',
                            $smsBody,
                            'failed',
                            null,
                            null,
                            $result['http_status'],
                            $result['error'] ?: $result['server_response'],
                            $date_created,
                            null
                        );
                    }
                }
                if ($sendSuccess && $successfulResult && $successfulProvider) {
                    $date_sent = date('Y-m-d H:i:s');
                    $decoded = json_decode($successfulResult['server_response'], true);
                    $sms_id = null;
                    if (is_array($decoded)) {
                        if (isset($decoded[0]['id'])) {
                            $sms_id = $decoded[0]['id'];
                        } elseif (isset($decoded['sid'])) {
                            $sms_id = $decoded['sid'];
                        }
                    }
                    // Log successful SMS
                    log_sms(
                        strtolower($successfulProvider['provider_name']),
                        $sms_id,
                        'outbound-api',
                        $patient_id,
                        $toNumber,
                        'Meditag',
                        $smsBody,
                        'sent',
                        null,
                        null,
                        $successfulResult['http_status'],
                        $successfulResult['error'] ?: $successfulResult['server_response'],
                        $date_created,
                        $date_sent
                    );
                    $smsSent[] = ['to' => $toNumber, 'status' => 'sent', 'sms_id' => $sms_id];
                } else {
                    $smsSent[] = ['to' => $toNumber, 'status' => 'failed', 'sms_id' => null];
                }
                error_log("Timing: SMS to $toNumber took " . (microtime(true) - $sms_start) . " seconds");
            }
            error_log("Timing: Full SMS sending loop took " . (microtime(true) - $start_time) . " seconds");
            $response->json(['success' => true, 'message' => 'Panic event stored successfully', 'event_id' => $event_id, 'sms_results' => $smsSent], 201);
        } catch (Exception $e) {
            $conn_panic->rollback();
            error_log("panic_events error: " . $e->getMessage());
            $response->json(['success' => false, 'message' => 'Internal server error'], 500);
        }
        break;
    case 'GET':
        $event_id = (int)$_GET['id'] ?? 0;
        if ($event_id === 0) {
            // List all events (endpoint 9)
            requireJWT(); // Requires JWT for list (admin/dashboard)
            $status = $_GET['status'] ?? null;
            $priority = $_GET['priority'] ?? null;
            $user_id_filter = (int)$_GET['user_id'] ?? 0;
            $limit = (int)$_GET['limit'] ?? 50;
            $limit = min(max($limit, 1), 100); // Clamp between 1-100
            $sql = "SELECT * FROM panic_button_events WHERE 1=1";
            $types = '';
            $params = [];
            if ($status) {
                $sql .= " AND status = ?";
                $types .= 's';
                $params[] = $status;
            }
            if ($priority) {
                $sql .= " AND priority = ?";
                $types .= 's';
                $params[] = $priority;
            }
            if ($user_id_filter > 0) {
                $sql .= " AND user_id = ?";
                $types .= 'i';
                $params[] = $user_id_filter;
            }
            $sql .= " ORDER BY created_at DESC LIMIT ?";
            $types .= 'i';
            $params[] = $limit;
            $stmt = prepareOrThrow($conn_panic, $sql, 'panic_events_list');
            if ($types) {
                $stmt->bind_param($types, ...$params);
            } else {
                // If no filters, just bind limit
                $stmt->bind_param('i', $limit);
            }
            executeOrThrow($stmt, 'panic_events_list');
            $result = $stmt->get_result();
            $events = [];
            while ($row = $result->fetch_assoc()) {
                $events[] = $row;
            }
            $stmt->close();
            $response->json(['events' => $events], 200);
        } else {
            // Single event by ID (endpoint 2)
            $key = $_GET['key'] ?? '';
            $isAuthenticated = false;
            $authUserId = null;
            // Optional JWT validation
            $header = $_SERVER['HTTP_AUTHORIZATION'] ?? '';
            if (preg_match('/Bearer\s+(\S+)/', $header, $matches)) {
                $payload = validateJWT($matches[1]);
                if ($payload) {
                    $isAuthenticated = true;
                    $authUserId = (int)$payload['userId'];
                }
            }
            try {
                if ($isAuthenticated) {
                    // Check if authenticated user owns the event
                    $stmt_owner = prepareOrThrow($conn_panic, "SELECT user_id FROM panic_button_events WHERE id = ?", 'panic_event_owner');
                    $stmt_owner->bind_param('i', $event_id);
                    executeOrThrow($stmt_owner, 'panic_event_owner');
                    $result_owner = $stmt_owner->get_result();
                    if ($result_owner->num_rows === 0) {
                        $response->json(['error' => 'Event not found'], 404);
                    }
                    $owner = $result_owner->fetch_assoc();
                    $stmt_owner->close();
                    if ($authUserId !== (int)$owner['user_id']) {
                        $response->json(['error' => 'Unauthorized: You do not own this event'], 403);
                    }
                } elseif (empty($key)) {
                    $response->json(['error' => 'Missing key (required without JWT)'], 400);
                } else {
                    // Validate key if no JWT
                    $stmt_key = prepareOrThrow($conn_panic, "SELECT `key` FROM event_updates WHERE event_id = ? LIMIT 1", 'event_updates_key');
                    $stmt_key->bind_param('i', $event_id);
                    executeOrThrow($stmt_key, 'event_updates_key');
                    $result_key = $stmt_key->get_result();
                    if ($result_key->num_rows === 0 || $result_key->fetch_assoc()['key'] !== $key) {
                        $response->json(['error' => 'Unauthorized: Invalid key'], 401);
                    }
                    $stmt_key->close();
                }
                $stmt_event = prepareOrThrow($conn_panic, "SELECT * FROM panic_button_events WHERE id = ?", 'panic_button_events_get');
                $stmt_event->bind_param('i', $event_id);
                executeOrThrow($stmt_event, 'panic_button_events_get');
                $result_event = $stmt_event->get_result();
                $event = $result_event->num_rows > 0 ? $result_event->fetch_assoc() : null;
                $stmt_event->close();
                if (!$event) {
                    $response->json(['error' => 'Event not found'], 404);
                }
                // Fetch region and patient name (regional routing)
                $conn_main = getDatabaseConnection('main', 'GBL');
                if (!$conn_main) {
                    throw new Exception("Main DB connection failed");
                }
                $stmt_region = prepareOrThrow($conn_main, "SELECT datacenter_region_code FROM user_datacenter_regions WHERE user_id = ? AND user_role_type = 'Patient' AND deleted_at IS NULL LIMIT 1", 'user_region_get');
                $stmt_region->bind_param('i', $event['user_id']);
                executeOrThrow($stmt_region, 'user_region_get');
                $result_region = $stmt_region->get_result();
                $region = $result_region->num_rows > 0 ? $result_region->fetch_assoc()['datacenter_region_code'] : 'GBL';
                $stmt_region->close();
                $conn_main_reg = getDatabaseConnection('main', $region);
                if (!$conn_main_reg) {
                    throw new Exception("Main regional DB connection failed for region: $region");
                }
                $stmt_user = prepareOrThrow($conn_main_reg, "SELECT first_name, last_name FROM users WHERE id = ? LIMIT 1", 'users_get');
                $stmt_user->bind_param('i', $event['user_id']);
                executeOrThrow($stmt_user, 'users_get');
                $result_user = $stmt_user->get_result();
                $user = $result_user->num_rows > 0 ? $result_user->fetch_assoc() : ['first_name' => 'Unknown', 'last_name' => ''];
                $stmt_user->close();
                $event['patient_name'] = $user['first_name'] . ' ' . $user['last_name'];
                // Fetch updates
                $stmt_updates = prepareOrThrow($conn_panic, "SELECT * FROM event_updates WHERE event_id = ? ORDER BY created_at DESC", 'event_updates_get');
                $stmt_updates->bind_param('i', $event_id);
                executeOrThrow($stmt_updates, 'event_updates_get');
                $result_updates = $stmt_updates->get_result();
                $updates = [];
                while ($row = $result_updates->fetch_assoc()) {
                    $updates[] = $row;
                }
                $stmt_updates->close();
                $event['updates'] = $updates;
                $response->json($event, 200);
            } catch (Exception $e) {
                error_log("panic_events GET error: " . $e->getMessage());
                $response->json(['error' => 'Internal server error'], 500);
            }
        }
        break;
    case 'PATCH':
        // Update event status/priority/description
        requireJWT(); // Admin check can be added via middleware/admin.php
        $event_id = (int)$_GET['id'] ?? 0;
        if (!$event_id) {
            $response->json(['error' => 'Missing event_id'], 400);
        }
        $status = isset($input['status']) ? $input['status'] : null;
        $priority = isset($input['priority']) ? $input['priority'] : null;
        $description = isset($input['description']) ? $input['description'] : null;
        $updates = [];
        $params = [];
        $types = '';
        if ($status) {
            $updates[] = 'status = ?';
            $types .= 's';
            $params[] = $status;
        }
        if ($priority) {
            $updates[] = 'priority = ?';
            $types .= 's';
            $params[] = $priority;
        }
        if ($description) {
            $updates[] = 'description = ?';
            $types .= 's';
            $params[] = $description;
        }
        if (empty($updates)) {
            $response->json(['error' => 'No fields to update'], 400);
        }
        $sql = "UPDATE panic_button_events SET " . implode(', ', $updates) . " WHERE id = ?";
        $types .= 'i';
        $params[] = $event_id;
        try {
            $stmt = prepareOrThrow($conn_panic, $sql, 'panic_button_events_patch');
            $stmt->bind_param($types, ...$params);
            executeOrThrow($stmt, 'panic_button_events_patch');
            $response->json(['success' => true, 'message' => 'Event updated'], 200);
            $stmt->close();
        } catch (Exception $e) {
            error_log("panic_events PATCH error: " . $e->getMessage());
            $response->json(['success' => false, 'message' => 'Internal server error'], 500);
        }
        break;
    default:
        $response->json(['error' => 'Method not allowed'], 405);
}
?>