<?php
// File name: qr_code.php
// File location: /home/apimedi1/public_html/api/v1/routes/medical/qr_code.php

require_once __DIR__ . '/../../middleware/jwt.php';
require_once __DIR__ . '/../../utils/request.php';
require_once __DIR__ . '/../../utils/region.php';
require_once __DIR__ . '/../../handlers/Response.php';

$userId = requireJWT();  // Authenticated user ID

$patientId = isset($_GET['patient_id']) ? (int)$_GET['patient_id'] : 0;
$qrCodeId = isset($_GET['qr_code_id']) ? (int)$_GET['qr_code_id'] : 0;

if ($patientId === 0 && $qrCodeId === 0) {
    (new Response())->json(['error' => 'Missing patient_id or qr_code_id'], 400);
}

$region = getPatientRegion($patientId ? $patientId : 0);  // Use patient_id if available

$connMainRegional = getDatabaseConnection('main', $region);
$connMainGlobal = getDatabaseConnection('main', 'GBL');  // Global main for non-replicated tables like user_qr_codes

if ($connMainRegional === null || $connMainRegional->connect_error) {
    (new Response())->json(['error' => 'Database connection failed for regional main'], 500);
}

if ($connMainGlobal === null || $connMainGlobal->connect_error) {
    (new Response())->json(['error' => 'Database connection failed for global main'], 500);
}

$method = $_SERVER['REQUEST_METHOD'];

// Define allowed QR code types matching the updated ENUM for validation
$allowedTypes = [
    'License Disk', 'Wristband', 'Keychain', 'Wallet Card', 'Mobile Wallet', 'ID Card', 'Smartwatch',
    'Medical Bracelet', 'Helmet Sticker', 'Pet Collar', 'Luggage Tag', 'Bike Frame', 'Sticker',
    'NFC Card', 'Airline Ticket', 'Bus Ticket', 'Train Ticket'
];

switch ($method) {
    case 'GET':
        // First, get user_id from patients in regional main
        $stmt = $connMainRegional->prepare("SELECT user_id FROM patients WHERE id = ?");
        if ($stmt === false) {
            (new Response())->json(['error' => 'Failed to prepare query for patients table: ' . $connMainRegional->error], 500);
        }
        $stmt->bind_param("i", $patientId);
        $stmt->execute();
        $result = $stmt->get_result();
        if (!$row = $result->fetch_assoc()) {
            (new Response())->json(['error' => 'Patient not found'], 404);
        }
        $userIdFromPatient = $row['user_id'];
        $stmt->close();

        // Now, fetch QR codes for that user_id from global main
        $stmt = $connMainGlobal->prepare("
            SELECT 
                qr_code_id, 
                qr_code_type AS type, 
                nickname, 
                status, 
                generated_timestamp AS generated_at, 
                path AS qr_image_url, 
                final_path AS final_image_url, 
                car_registration_number
            FROM user_qr_codes 
            WHERE user_id = ?
        ");
        if ($stmt === false) {
            (new Response())->json(['error' => 'Failed to prepare query for user_qr_codes table: ' . $connMainGlobal->error], 500);
        }
        $stmt->bind_param("i", $userIdFromPatient);
        $stmt->execute();
        $result = $stmt->get_result();

        $qrCodes = [];
        while ($row = $result->fetch_assoc()) {
            // Filter out null values if desired
            $qrCodes[] = array_filter($row, function($value) {
                return $value !== null;
            });
        }

        if (empty($qrCodes)) {
            (new Response())->json(['error' => 'No QR codes found for this patient'], 404);
        }

        (new Response())->json($qrCodes);
        $stmt->close();
        break;

    case 'POST':
        // Generate new QR code
        $input = getJsonInput();
        $type = $input['type'] ?? '';
        $nickname = $input['nickname'] ?? '';
        $carReg = $input['car_registration_number'] ?? null;

        if (empty($type) || empty($nickname)) {
            (new Response())->json(['error' => 'Missing required fields: type and nickname'], 400);
        }
        if (!in_array($type, $allowedTypes)) {
            (new Response())->json(['error' => 'Invalid QR code type. Allowed types: ' . implode(', ', $allowedTypes)], 400);
        }

        // Get patient's user_id from regional main
        $stmt = $connMainRegional->prepare("SELECT user_id FROM patients WHERE id = ?");
        if ($stmt === false) {
            (new Response())->json(['error' => 'Failed to prepare query for patients table: ' . $connMainRegional->error], 500);
        }
        $stmt->bind_param("i", $patientId);
        $stmt->execute();
        $result = $stmt->get_result();
        if (!$row = $result->fetch_assoc()) {
            (new Response())->json(['error' => 'Patient not found'], 404);
        }
        $patientUserId = $row['user_id'];
        $stmt->close();

        // Generate unique key
        $key = bin2hex(random_bytes(16));

        // Insert into user_qr_codes in global main
        $insertStmt = $connMainGlobal->prepare("
            INSERT INTO user_qr_codes 
            (user_id, `key`, service, status, generated_timestamp, qr_code_type, car_registration_number, nickname) 
            VALUES (?, ?, 'EMI', 'active', NOW(), ?, ?, ?)
        ");
        if ($insertStmt === false) {
            (new Response())->json(['error' => 'Failed to prepare insert query for user_qr_codes: ' . $connMainGlobal->error], 500);
        }
        $insertStmt->bind_param("issss", $patientUserId, $key, $type, $carReg, $nickname);
        $insertStmt->execute();
        $qrId = $connMainGlobal->insert_id;
        $insertStmt->close();

        // Generate QR data URL
        $qrData = BASE_URL . 'scan_tag.php?qr_code_id=' . $qrId . '&key=' . $key;

        // Fetch QR image from external API
        $qrApiUrl = 'https://api.qrserver.com/v1/create-qr-code/?size=300x300&data=' . urlencode($qrData);
        $qrContent = file_get_contents($qrApiUrl);

        if ($qrContent === false) {
            (new Response())->json(['error' => 'Failed to generate QR code image'], 500);
        }

        $rawUrl = '';  // Placeholder for uploaded raw QR URL
        $finalUrl = '';  // Placeholder for final (superimposed) URL

        // For License Disk, superimpose using GD
        if ($type === 'License Disk') {
            // Load template (use relative path)
            $templatePath = __DIR__ . '/../../../../mnt/data/Meditag_Car_EMI_Tag.png';
            $background = imagecreatefrompng($templatePath);
            $qrImg = imagecreatefromstring($qrContent);

            if ($background && $qrImg) {
                imagecopyresampled($background, $qrImg, 515, 390, 0, 0, 220, 220, imagesx($qrImg), imagesy($qrImg));
                ob_start();
                imagepng($background);
                $finalContent = ob_get_clean();
                imagedestroy($background);
                imagedestroy($qrImg);

                // Here, upload $finalContent if needed, or return base64
                $finalUrl = 'data:image/png;base64,' . base64_encode($finalContent);
            } else {
                (new Response())->json(['error' => 'Failed to process License Disk image'], 500);
            }
        } else {
            $rawUrl = 'data:image/png;base64,' . base64_encode($qrContent);
        }

        // Update paths in DB (if uploading, update with URLs)
        $updateStmt = $connMainGlobal->prepare("UPDATE user_qr_codes SET path = ?, final_path = ?, qr_code_data = ? WHERE qr_code_id = ?");
        if ($updateStmt === false) {
            (new Response())->json(['error' => 'Failed to prepare update query for user_qr_codes: ' . $connMainGlobal->error], 500);
        }
        $updateStmt->bind_param("sssi", $rawUrl, $finalUrl, $qrData, $qrId);
        $updateStmt->execute();
        $updateStmt->close();

        (new Response())->json([
            'qr_code_id' => $qrId,
            'type' => $type,
            'nickname' => $nickname,
            'qr_image_url' => $rawUrl,
            'final_image_url' => $finalUrl
        ], 201);
        break;

    case 'DELETE':
        // Revoke QR code
        $stmt = $connMainGlobal->prepare("UPDATE user_qr_codes SET status = 'revoked' WHERE qr_code_id = ?");
        if ($stmt === false) {
            (new Response())->json(['error' => 'Failed to prepare delete query for user_qr_codes: ' . $connMainGlobal->error], 500);
        }
        $stmt->bind_param("i", $qrCodeId);
        if ($stmt->execute() && $stmt->affected_rows > 0) {
            http_response_code(204);
            exit;
        } else {
            (new Response())->json(['error' => 'QR code not found or already revoked'], 404);
        }
        $stmt->close();
        break;

    default:
        (new Response())->json(['error' => 'Method not allowed'], 405);
}
?>