<?php
// File name: logout.php
// File location: /home/apimedi1/public_html/api/v1/routes/auth/logout.php

// Add this line to explicitly require the JWT middleware
// (since it's in a subfolder, and to ensure it's loaded)
require_once __DIR__ . '/../../middleware/jwt.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    (new Response)->json(['error' => 'Method Not Allowed'], 405);
}

// Require valid JWT – we need the user_id from the token
$userId = requireJWT(); // Sets $GLOBALS['auth_user_id'] and returns userId

// Optional: allow a method to be sent (e.g., "manual", "timeout", "session_expired")
$data = getJsonInput();
$logoutMethod = $data['logout_method'] ?? 'manual';
$logoutMethod = substr(trim($logoutMethod), 0, 50); // safety limit
$sessionId = $data['session_id'] ?? null; // Optional: target specific session

// Get the token from header
$header = $_SERVER['HTTP_AUTHORIZATION'] ?? '';
if (preg_match('/Bearer\s+(\S+)/', $header, $m)) {
    $token = $m[1];
} else {
    $token = '';
}

// Parse exp from token (base64 decode payload)
$parts = explode('.', $token);
$payloadJson = base64_decode($parts[1] ?? '');
$payload = json_decode($payloadJson, true);
$expiry = $payload['exp'] ?? time() + 3600; // Default to 1 hour if parse fails

// Find the most recent login entry for this user that has no logout_time yet
$connAudit = getConnAudit();
if (!$connAudit) {
    (new Response)->json(['error' => 'Audit database unavailable'], 500);
}

if ($sessionId) {
    // Target specific session_id if provided
    $stmt = $connAudit->prepare("
        SELECT session_id 
        FROM user_login_logs 
        WHERE user_id = ? 
          AND session_id = ? 
          AND logout_time IS NULL
    ");
    $stmt->bind_param("is", $userId, $sessionId);
} else {
    // Fall back to most recent if no session_id
    $stmt = $connAudit->prepare("
        SELECT session_id 
        FROM user_login_logs 
        WHERE user_id = ? 
          AND logout_time IS NULL 
        ORDER BY login_time DESC 
        LIMIT 1
    ");
    $stmt->bind_param("i", $userId);
}

if (!$stmt) {
    logError("Logout query prepare failed: " . $connAudit->error); // Log to custom log
    (new Response)->json(['error' => 'Internal server error'], 500);
}

$stmt->execute();
$result = $stmt->get_result();

if ($row = $result->fetch_assoc()) {
    $sessionId = $row['session_id'];

    $update = $connAudit->prepare("
        UPDATE user_login_logs 
        SET logout_time = NOW(), 
            logout_method = ? 
        WHERE session_id = ? 
          AND user_id = ?
    ");
    if (!$update) {
        logError("Logout update prepare failed: " . $connAudit->error);
        (new Response)->json(['error' => 'Internal server error'], 500);
    }

    $update->bind_param("ssi", $logoutMethod, $sessionId, $userId);
    $update->execute();
    $update->close();

    // Insert token into revoked_tokens (in core DB)
    $connMain = getConnMain();
    $revokeStmt = $connMain->prepare("
        INSERT INTO revoked_tokens 
        (token, user_id, revoked_at, expiry)
        VALUES (?, ?, NOW(), ?)
    ");
    if (!$revokeStmt) {
        logError("Revoke token prepare failed: " . $connMain->error);
        (new Response)->json(['error' => 'Internal server error'], 500);
    }

    $revokeStmt->bind_param("sii", $token, $userId, $expiry);
    $revokeStmt->execute();
    $revokeStmt->close();

    // Success response if active session was logged out
    (new Response)->json([
        'message' => 'Logged out successfully',
        'logout_method' => $logoutMethod
    ], 200);
} else {
    // No active session found – user was already logged out
    (new Response)->json([
        'message' => 'No active session found – already logged out',
        'logout_method' => $logoutMethod
    ], 200);
}

$stmt->close();
?>